﻿using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.Configuration.Json;
using System;
using System.IO;
using System.Text.RegularExpressions;

namespace Edu.Common
{
    /// <summary>
    /// 全局配置
    /// </summary>
    public class Config
    {
        /// <summary>
        /// 配置文件注入
        /// </summary>
        public readonly static IConfiguration Configuration;
        static Config()
        {
            try
            {
                Configuration = new ConfigurationBuilder()
                .Add(new JsonConfigurationSource { Path = "appsettings.json", ReloadOnChange = true })
                .Build();
            }
            catch (Exception ex)
            {
                Common.Plugin.LogHelper.Write(ex, "Config");
            }
        }

        /// <summary>
        /// 获取连接字符串
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public static string GetConnectionString(string key)
        {
            try
            {
                return Configuration.GetConnectionString(key);
            }
            catch
            {
                return "";
            }
        }


        /// <summary>
        /// 读取配置文件key的值
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public static string ReadConfigKey(string key, string subKey = "")
        {
            string valueStr = "";
            try
            {
                if (!string.IsNullOrWhiteSpace(subKey))
                {
                    valueStr = Configuration.GetSection(key)[subKey].ToString();
                }
                else
                {
                    valueStr = Configuration.GetSection(key).Value;
                }
            }
            catch
            {
            }
            return valueStr;
        }
 
        /// <summary>
        /// 异常日志保存路径
        /// </summary>
        public static string LogPath
        {
            get
            {
                string logPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "log/error");
                return logPath.ToString();
            }
        }

        /// <summary>
        /// 打印日志保存路径
        /// </summary>
        public static string InofLogPath
        {
            get
            {
                string logPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "log/info");
                return logPath.ToString();
            }
        }

        /// <summary>
        /// 接口请求日志保存路径
        /// </summary>
        public static string RequestLogPath
        {
            get
            {
                string logPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "log/request");
                return logPath.ToString();
            }
        }

        /// <summary>
        /// JWT加密秘钥
        /// </summary>
        public static string JwtSecretKey
        {
            get
            {
                return ReadConfigKey("JwtSecretKey");
            }
        }

        /// <summary>
        /// 过期时间，单位为秒
        /// </summary>
        public static int JwtExpirTime
        {
            get
            {
                Int32.TryParse(ReadConfigKey("JwtExpirTime"), out int JwtExpirTime);
                return JwtExpirTime;
            }
        }
 

        /// <summary>
        /// 阿里云oss域名
        /// </summary>
        public static string GetOssFileUrl
        {
            get
            {
                return ReadConfigKey("ViewFileSiteUrl");
            }
        }

        /// <summary>
        /// 获取Oss文件全路径
        /// </summary>
        /// <param name="path">文件相对路径</param>
        /// <returns></returns>
        public static string GetFileUrl(string path)
        {
            if (string.IsNullOrWhiteSpace(path))
            {
                return null;
            }
            else
            {
                string p = @"(http|https)://(?<domain>[^(:|/]*)";
                Regex reg = new Regex(p, RegexOptions.IgnoreCase);
                string value = reg.Match(path).Value;
                if (!string.IsNullOrEmpty(value))
                {
                    return path;
                }
                else
                {
                    return GetOssFileUrl + path;
                }
            }
        }

        /// <summary>
        /// 获取文件相对路径
        /// </summary>
        public static string GetFilePath(string path)
        {
            if (!string.IsNullOrWhiteSpace(path))
            {
                string p = @"(http|https)://(?<domain>[^(:|/]*)";
                Regex reg = new Regex(p, RegexOptions.IgnoreCase);
                string value = reg.Match(path).Value;
                if (!string.IsNullOrEmpty(value))
                {
                    Uri url = new Uri(path);
                    return url.AbsolutePath;
                }
                else
                {
                    return path;
                }
            }
            else
            {
                return path;
            }
        }

        /// <summary>
        /// 上传站点地址
        /// </summary>
        public static string UploadSiteUrl
        {
            get
            {
                return ReadConfigKey("UploadSiteUrl");
            }
        }

        /// <summary>
        /// 上传文件预览站点
        /// </summary>
        public static string ViewFileSiteUrl
        {
            get
            {
                return ReadConfigKey("ViewFileSiteUrl");
            }
        }

        /// <summary>
        /// 获取虚拟目录
        /// </summary>
        public static string VirtualDirectory
        {
            get
            {
                return ReadConfigKey("VirtualDirectory");
            }
        }

        /// <summary>
        /// Mongo连接地址
        /// </summary>
        public static string Mongo
        {
            get
            {
                return ReadConfigKey("Mongo");
            }
        }

        /// <summary>
        /// 获取MongoDBName
        /// </summary>
        public static string MongoDBName
        {
            get
            {
                return ReadConfigKey("MongoDBName");
            }
        }

        /// <summary>
        /// 默认密码
        /// </summary>
        public static string DefaultPwd
        {
            get { return "123456"; }
        }
    }
}