﻿using Edu.Common.Enum;
using Edu.Model.ViewModel.Course;
using Edu.Repository.Course;
using System.Collections.Generic;
using System.Linq;
using VT.FW.DB;

namespace Edu.Module.Course
{
    /// <summary>
    /// 课程分类处理类
    /// </summary>
    public class CourseCategoryModule
    {
        /// <summary>
        /// 课程分类仓储对象
        /// </summary>
        private readonly RB_Course_CategoryRepository categoryRepository = new RB_Course_CategoryRepository();

        /// <summary>
        /// 获取课程分类列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<RB_Course_Category_ViewModel> GetCourseCategoryListModule(RB_Course_Category_ViewModel query)
        {
            return categoryRepository.GetCourseCategoryListRepository(query);
        }

        /// <summary>
        /// 获取课程分类分页列表
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <param name="pageSize"></param>
        /// <param name="rowsCount"></param>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<RB_Course_Category_ViewModel> GetCourseCategoryPageListModule(int pageIndex, int pageSize, out long rowsCount, RB_Course_Category_ViewModel query)
        {
            return categoryRepository.GetCourseCategoryPageListRepository(pageIndex, pageSize, out rowsCount, query);
        }

        /// <summary>
        /// 获取当前和当前所有下级分类列表
        /// </summary>
        /// <param name="cateIds">事例(1,5)</param>
        /// <returns></returns>
        public List<RB_Course_Category_ViewModel> GetChildCategoryListModule(string cateIds)
        {
            return categoryRepository.GetChildCategoryListRepository(cateIds);
        }

        /// <summary>
        /// 获取当前和当前所有下级分类编号
        /// </summary>
        /// <param name="cateIds">事例(1,5)</param>
        /// <returns></returns>
        public string GetChildCategoryStringModule(string cateIds)
        {
            string result = "";
            var list = GetChildCategoryListModule(cateIds);
            if (list != null && list.Count > 0)
            {
                result = string.Join(",", list.Select(qitem => qitem.CateId));
            }
            return result;
        }

        /// <summary>
        /// 根据分类编号获取课程分类实体
        /// </summary>
        /// <param name="CateId"></param>
        /// <returns></returns>
        public RB_Course_Category_ViewModel GetCourseCategoryModule(object CateId)
        {
            return categoryRepository.GetEntity<RB_Course_Category_ViewModel>(CateId);
        }

        /// <summary>
        /// 获取课程分类树形列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<CourseCategoryTree_ViewModel> GetCourseCategoryTreeListModule(RB_Course_Category_ViewModel query)
        {
            List<CourseCategoryTree_ViewModel> treeList = new List<CourseCategoryTree_ViewModel>();
            var list = GetCourseCategoryListModule(query);
            if (list != null && list.Count > 0)
            {
                var firstList = list.Where(qitem => qitem.ParentId == 0).ToList();
                if (firstList != null && firstList.Count > 0)
                {
                    foreach (var fItem in firstList)
                    {
                        CourseCategoryTree_ViewModel tModel = new CourseCategoryTree_ViewModel()
                        {
                            CateId = fItem.CateId,
                            CateName = fItem.CateName,
                            ParentId = fItem.ParentId,
                            ChildList = new List<CourseCategoryTree_ViewModel>()
                        };
                        tModel.ChildList = GetCategoryTreeList(fItem.CateId, list);
                        treeList.Add(tModel);
                    }
                }
            }
            return treeList;
        }

        /// <summary>
        /// 递归生成树形结构
        /// </summary>
        /// <param name="parentId">父节点编号</param>
        /// <param name="sourceList">数据源列表</param>
        private List<CourseCategoryTree_ViewModel> GetCategoryTreeList(int parentId, List<RB_Course_Category_ViewModel> sourceList)
        {
            List<CourseCategoryTree_ViewModel> treeList = new List<CourseCategoryTree_ViewModel>();
            foreach (var item in sourceList.Where(qitem => qitem.ParentId == parentId))
            {
                CourseCategoryTree_ViewModel model = new CourseCategoryTree_ViewModel()
                {
                    CateId = item.CateId,
                    CateName = item.CateName,
                    ParentId = item.ParentId,
                    ChildList = new List<CourseCategoryTree_ViewModel>(),
                };
                model.ChildList = GetCategoryTreeList(item.CateId, sourceList);
                treeList.Add(model);
            }
            return treeList;
        }

        /// <summary>
        /// 新增修改课程分类
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        public bool SetCourseCategoryModule(RB_Course_Category_ViewModel model)
        {
            bool flag;
            if (model.CateId > 0)
            {
                Dictionary<string, object> fileds = new Dictionary<string, object>()
                {
                    {nameof(RB_Course_Category_ViewModel.ParentId),model.ParentId },
                    {nameof(RB_Course_Category_ViewModel.CateName),model.CateName.Trim() },
                    {nameof(RB_Course_Category_ViewModel.SortNum),model.SortNum },
                    {nameof(RB_Course_Category_ViewModel.UpdateBy),model.UpdateBy },
                    {nameof(RB_Course_Category_ViewModel.UpdateTime),model.UpdateTime },
                };
                flag = categoryRepository.Update(fileds, new WhereHelper(nameof(RB_Course_Category_ViewModel.CateId), model.CateId));
            }
            else
            {
                var newId = categoryRepository.Insert(model);
                model.CateId = newId;
                flag = newId > 0;
            }
            return flag;
        }

        /// <summary>
        /// 根据分类编号删除课程分类
        /// </summary>
        /// <param name="CateId"></param>
        /// <returns></returns>
        public bool RemoveCourseCategoryModule(int CateId)
        {
            Dictionary<string, object> fileds = new Dictionary<string, object>()
            {
                {nameof(RB_Course_Category_ViewModel.Status),(int)DateStateEnum.Delete },
            };
            bool flag = categoryRepository.Update(fileds, new WhereHelper(nameof(RB_Course_Category_ViewModel.CateId), CateId));
            return flag;
        }
    }
}