﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Mall.Common.Plugin
{
    /// <summary>
    /// 枚举帮助类
    /// </summary>
    public static class EnumHelper
    {
        /// <summary>
        /// 获取枚举值集合
        /// </summary>
        /// <param name="enumType"></param>
        /// <returns></returns>
        public static Dictionary<string, string> GetEnumList(Type enumType)
        {
            Dictionary<string, string> dic = new Dictionary<string, string>();
            FieldInfo[] fieldinfos = enumType.GetFields();
            foreach (FieldInfo field in fieldinfos)
            {
                if (field.FieldType.IsEnum)
                {
                    Object[] objs = field.GetCustomAttributes(typeof(EnumFieldAttribute), false);
                    dic.Add(((EnumFieldAttribute)objs[0]).Name, field.GetRawConstantValue().ToString());
                }
            }
            return dic;
        }

        /// <summary>
        /// 获取枚举列表
        /// </summary>
        /// <param name="enumType"></param>
        /// <returns></returns>
        public static List<EnumItem> EnumToList(Type enumType)
        {
            List<EnumItem> list = new List<EnumItem>();
            FieldInfo[] fieldinfos = enumType.GetFields();
            foreach (FieldInfo field in fieldinfos)
            {
                if (field.FieldType.IsEnum)
                {
                    Object[] objs = field.GetCustomAttributes(typeof(EnumFieldAttribute), false);
                    Int32.TryParse(field.GetRawConstantValue().ToString(), out int Id);
                    list.Add(new EnumItem()
                    {
                        Id = Id,
                        Name = ((EnumFieldAttribute)objs[0]).Name
                    });
                }
            }
            return list;
        }

        /// <summary>
        /// 获取枚举值的名称
        /// </summary>
        /// <param name="e"></param>
        /// <returns></returns>
        public static string GetEnumName(object e)
        {
            if (e == null)
            {
                return "";
            }
            //获取字段信息
            System.Reflection.FieldInfo[] ms = e.GetType().GetFields();

            Type t = e.GetType();
            foreach (System.Reflection.FieldInfo f in ms)
            {
                //判断名称是否相等
                if (f.Name != e.ToString()) continue;

                //反射出自定义属性
                object[] attrs = f.GetCustomAttributes(typeof(EnumFieldAttribute), true);
                if (attrs.Length > 0)
                {
                    EnumFieldAttribute attr = attrs[0] as EnumFieldAttribute;
                    return attr.Name;
                }
            }
            return "";
        }

        /// <summary>
        /// 获取枚举特性描述值
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static string GetEnumName(this System.Enum obj)
        {
            if (obj == null)
            {
                return "";
            }
            try
            {
                Type type = obj.GetType();
                FieldInfo fild = type.GetField(obj.ToString());
                if (fild.IsDefined(typeof(EnumFieldAttribute), true))
                {
                    EnumFieldAttribute enumField = (EnumFieldAttribute)fild.GetCustomAttribute(typeof(EnumFieldAttribute), true);
                    return enumField.Name;
                }
                else
                {
                    return "";
                }
            }
            catch
            {
                return "";
            }
        }

        /// <summary>
        /// 获取枚举成员的值(this是扩展方法的标志)
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static int ToInt(this System.Enum obj)
        {
            return Convert.ToInt32(obj);
        }

        /// <summary>
        /// 将值转成枚举
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="obj"></param>
        /// <returns></returns>
        public static T ToEnum<T>(this string obj) where T : struct
        {
            if (string.IsNullOrEmpty(obj))
            {
                return default(T);
            }
            try
            {
                return (T)System.Enum.Parse(typeof(T), obj, true);
            }
            catch (Exception)
            {
                return default(T);
            }
        }
    }

    /// <summary>
    /// 枚举特性标签
    /// </summary>
    public class EnumFieldAttribute : System.Attribute
    {
        /// <summary>
        /// 名称
        /// </summary>
        public string Name { get; set; }
        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="name"></param>
        public EnumFieldAttribute(string name)
        {
            this.Name = name;
        }
    }

    /// <summary>
    /// 枚举项
    /// </summary>
    public class EnumItem
    {
        /// <summary>
        /// 编号
        /// </summary>
        public int Id { get; set; }

        /// <summary>
        /// 枚举名称
        /// </summary>
        public string Name { get; set; }
    }
}
