﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Mall.Common;
using Mall.Common.API;
using Mall.Common.Plugin;
using Mall.Model.Entity.Product;
using Mall.Model.Extend.Product;
using Mall.Repository;
using Mall.Repository.Product;

namespace Mall.Module.Product
{
    /// <summary>
    /// 商品处理层
    /// </summary>
    public class ProductModule
    {
        /// <summary>
        /// 素材分组
        /// </summary>
        private readonly RB_Material_GroupRepository material_GroupRepository = new RB_Material_GroupRepository();
        /// <summary>
        /// 素材管理
        /// </summary>
        private readonly RB_Material_InfoRepository material_InfoRepository = new RB_Material_InfoRepository();
        /// <summary>
        /// 商品分类
        /// </summary>
        private readonly RB_Product_CategoryRepository product_CategoryRepository = new RB_Product_CategoryRepository();
        /// <summary>
        /// 商品分类样式
        /// </summary>
        private readonly RB_Product_CategoryStyleRepository product_CategoryStyleRepository = new RB_Product_CategoryStyleRepository();


        #region 素材管理
        /// <summary>
        /// 获取素材分组列表
        /// </summary>
        /// <param name="dmodel"></param>
        /// <returns></returns>
        public List<RB_Material_Group_Extend> GetMaterialGroupList(RB_Material_Group_Extend dmodel) {
            var list= material_GroupRepository.GetList(dmodel);
            return list;
        }


        /// <summary>
        /// 获取素材分组分页列表
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <param name="pageSize"></param>
        /// <param name="count"></param>
        /// <param name="dmodel"></param>
        /// <returns></returns>
        public List<RB_Material_Info_Extend> GetMaterialInfoPageList(int pageIndex, int pageSize, out long count, RB_Material_Info_Extend dmodel)
        {
            var list = material_InfoRepository.GetPageList(pageIndex, pageSize, out count, dmodel);
            return list;
        }

        /// <summary>
        /// 新增分组
        /// </summary>
        /// <param name="demodel"></param>
        /// <returns></returns>
        public bool SetMaterialGroupInfo(RB_Material_Group_Extend demodel)
        {
            if (demodel.Id > 0)
            {
                Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                    { nameof(RB_Material_Group.Name),demodel.Name},
                    { nameof(RB_Material_Group.Sort),demodel.Sort},
                    { nameof(RB_Material_Group.UpdateDate),demodel.UpdateDate},
                };
                List<WhereHelper> wheres = new List<WhereHelper>() {
                    new WhereHelper(){
                     FiledName=nameof(RB_Material_Group.Id),
                      FiledValue=demodel.Id,
                       OperatorEnum=OperatorEnum.Equal
                    }
                };
                return material_GroupRepository.Update(keyValues, wheres);
            }
            else {
                return material_GroupRepository.Insert(demodel) > 0;
            }
        }

        /// <summary>
        /// 设置素材信息
        /// </summary>
        /// <param name="demodel"></param>
        /// <returns></returns>
        public bool SetMaterialInfo(RB_Material_Info_Extend demodel)
        {
            if (demodel.Id > 0)
            {
                Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                    { nameof(RB_Material_Info.Name),demodel.Name},
                    { nameof(RB_Material_Info.UpdateDate),demodel.UpdateDate},
                };
                List<WhereHelper> wheres = new List<WhereHelper>() {
                    new WhereHelper(){
                     FiledName=nameof(RB_Material_Group.Id),
                      FiledValue=demodel.Id,
                       OperatorEnum=OperatorEnum.Equal
                    }
                };
                return material_InfoRepository.Update(keyValues, wheres);
            }
            else {
                return material_InfoRepository.Insert(demodel) > 0;
            }
        }

        /// <summary>
        /// 设置分组回收
        /// </summary>
        /// <param name="groupId"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool SetMaterialGroupRecycled(int groupId, string uid, int mallBaseId)
        {
            var gmodel = material_GroupRepository.GetEntity(groupId);
            if (gmodel == null) {
                return false;
            }
            Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                { nameof(RB_Material_Group.Recycled),2},
                { nameof(RB_Material_Group.UpdateDate),DateTime.Now},
            };
            List<WhereHelper> wheres = new List<WhereHelper>() {
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.Id),
                    FiledValue=groupId,
                    OperatorEnum=OperatorEnum.Equal
                },
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.TenantId),
                    FiledValue=Convert.ToInt32(uid),
                    OperatorEnum=OperatorEnum.Equal
                },
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.MallBaseId),
                    FiledValue=mallBaseId,
                    OperatorEnum=OperatorEnum.Equal
                },
            };
            bool flag = material_GroupRepository.Update(keyValues, wheres);
            if (flag) {
                //分组下所有素材全回收
                material_InfoRepository.SetMaterialInfoRecycled(groupId, Convert.ToInt32(uid), mallBaseId);
            }
            return flag;
        }

        /// <summary>
        /// 删除分组
        /// </summary>
        /// <param name="groupId"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool DelMaterialGroupInfo(int groupId, string uid, int mallBaseId)
        {
            var gmodel = material_GroupRepository.GetEntity(groupId);
            if (gmodel == null)
            {
                return false;
            }
            if (gmodel.Recycled != 2) {
                return false;
            }
            Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                { nameof(RB_Material_Group.Status),1},
                { nameof(RB_Material_Group.UpdateDate),DateTime.Now},
            };
            List<WhereHelper> wheres = new List<WhereHelper>() {
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.Id),
                    FiledValue=groupId,
                    OperatorEnum=OperatorEnum.Equal
                },
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.TenantId),
                    FiledValue=Convert.ToInt32(uid),
                    OperatorEnum=OperatorEnum.Equal
                },
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.MallBaseId),
                    FiledValue=mallBaseId,
                    OperatorEnum=OperatorEnum.Equal
                },
            };
            bool flag = material_GroupRepository.Update(keyValues, wheres);
            if (flag) {
                //删除分组下所有的素材
                material_InfoRepository.DelMaterialInfo("", uid, mallBaseId, groupId);
            }
            return flag;
        }

        /// <summary>
        /// 设置分组还原
        /// </summary>
        /// <param name="groupId"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool SetMaterialGroupRestore(int groupId, string uid, int mallBaseId)
        {
            var gmodel = material_GroupRepository.GetEntity(groupId);
            if (gmodel == null)
            {
                return false;
            }
            if (gmodel.Recycled != 2)
            {
                return false;
            }
            Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                { nameof(RB_Material_Group.Recycled),1},
                { nameof(RB_Material_Group.UpdateDate),DateTime.Now},
            };
            List<WhereHelper> wheres = new List<WhereHelper>() {
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.Id),
                    FiledValue=groupId,
                    OperatorEnum=OperatorEnum.Equal
                },
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.TenantId),
                    FiledValue=Convert.ToInt32(uid),
                    OperatorEnum=OperatorEnum.Equal
                },
                new WhereHelper(){
                    FiledName=nameof(RB_Material_Group.MallBaseId),
                    FiledValue=mallBaseId,
                    OperatorEnum=OperatorEnum.Equal
                },
            };
            return material_GroupRepository.Update(keyValues, wheres);
        }

        /// <summary>
        /// 设置素材回收
        /// </summary>
        /// <param name="materialIds"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool SetMaterialInfoRecycled(string materialIds, string uid, int mallBaseId)
        {
            bool flag = material_InfoRepository.SetMaterialListRecycled(materialIds, uid, mallBaseId);
            return flag;
        }

        /// <summary>
        /// 删除素材
        /// </summary>
        /// <param name="materialIds"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool DelMaterialInfo(string materialIds, string uid, int mallBaseId)
        {
            bool flag = material_InfoRepository.DelMaterialInfo(materialIds, uid, mallBaseId);
            return flag;
        }

        /// <summary>
        /// 设置素材还原
        /// </summary>
        /// <param name="materialIds"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool SetMaterialInfoRestore(string materialIds, string uid, int mallBaseId)
        {
            var trans = material_InfoRepository.DbTransaction;
            try
            {
                var list = material_InfoRepository.GetList(new RB_Material_Info_Extend() { MaterialIds = materialIds, TenantId = Convert.ToInt32(uid), MallBaseId = mallBaseId });
                if (list.Any())
                {
                    //查询出所有的分组
                    var glist = material_GroupRepository.GetList(new RB_Material_Group_Extend() { GroupIds = string.Join(",", list.Select(x => x.GroupId).Distinct()), TenantId = Convert.ToInt32(uid), MallBaseId = mallBaseId });
                    if (glist.Where(x => x.Recycled == 2).Any())
                    {
                        foreach (var item in glist.Where(x => x.Recycled == 2)) {
                            Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                                { nameof(RB_Material_Group.Recycled),1},
                                { nameof(RB_Material_Group.UpdateDate),DateTime.Now},
                            };
                            List<WhereHelper> wheres = new List<WhereHelper>() {
                                new WhereHelper(){
                                    FiledName=nameof(RB_Material_Group.Id),
                                    FiledValue=item.Id,
                                    OperatorEnum=OperatorEnum.Equal
                                },
                                new WhereHelper(){
                                    FiledName=nameof(RB_Material_Group.TenantId),
                                    FiledValue=Convert.ToInt32(uid),
                                    OperatorEnum=OperatorEnum.Equal
                                },
                                new WhereHelper(){
                                    FiledName=nameof(RB_Material_Group.MallBaseId),
                                    FiledValue=mallBaseId,
                                    OperatorEnum=OperatorEnum.Equal
                                },
                            };
                            material_GroupRepository.Update(keyValues, wheres, trans);
                        }
                    }
                    foreach (var item in list) {
                        Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                            { nameof(RB_Material_Info.Recycled),1},
                            { nameof(RB_Material_Info.UpdateDate),DateTime.Now},
                        };
                        List<WhereHelper> wheres = new List<WhereHelper>() {
                            new WhereHelper(){
                                FiledName=nameof(RB_Material_Info.Id),
                                FiledValue=item.Id,
                                OperatorEnum=OperatorEnum.Equal
                            },
                            new WhereHelper(){
                                FiledName=nameof(RB_Material_Info.TenantId),
                                FiledValue=Convert.ToInt32(uid),
                                OperatorEnum=OperatorEnum.Equal
                            },
                            new WhereHelper(){
                                FiledName=nameof(RB_Material_Info.MallBaseId),
                                FiledValue=mallBaseId,
                                OperatorEnum=OperatorEnum.Equal
                            },
                        };
                        material_InfoRepository.Update(keyValues, wheres, trans);
                    }
                }
                material_InfoRepository.DBSession.Commit();
                return true;
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, "SetMaterialInfoRestore");
                material_InfoRepository.DBSession.Rollback();
                return false;
            }
        }

        /// <summary>
        /// 设置分组转移
        /// </summary>
        /// <param name="materialIds"></param>
        /// <param name="groupId"></param>
        /// <param name="Type"></param>
        /// <param name="uid"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool SetMaterialInfoTransfer(string materialIds, int groupId, int Type, string uid, int mallBaseId)
        {
            var gmodel = material_GroupRepository.GetEntity(groupId);
            if (gmodel == null) {
                return false;
            }
            if (gmodel.Status == 1 || gmodel.Recycled == 2 || gmodel.Type != Type)
            {
                return false;
            }
            return material_InfoRepository.SetMaterialInfoTransfer(materialIds, groupId, uid, mallBaseId);
        }

        #endregion

        #region 商品分类

        /// <summary>
        /// 分页列表
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <param name="pageSize"></param>
        /// <param name="count"></param>
        /// <param name="demodel"></param>
        /// <returns></returns>
        public List<RB_Product_Category_Extend> GetProductCategoryPageList(int pageIndex, int pageSize, out long count, RB_Product_Category_Extend demodel)
        {
            var list = product_CategoryRepository.GetPageList(pageIndex, pageSize, out count, demodel);
            if (list.Any()) {
                List<int> ImgIds = new List<int>();
                ImgIds.AddRange(list.Select(x => x.Icon ?? 0).Distinct());
                ImgIds.AddRange(list.Select(x => x.BigIcon ?? 0).Distinct());
                ImgIds.AddRange(list.Select(x => x.AdvertisingImage ?? 0).Distinct());
                List<RB_Material_Info_Extend> mlist = new List<RB_Material_Info_Extend>();
                if (ImgIds.Any()) {
                    mlist = material_InfoRepository.GetList(new RB_Material_Info_Extend() { MaterialIds = string.Join(",", ImgIds), TenantId = demodel.TenantId, MallBaseId = demodel.MallBaseId });
                }

                foreach (var item in list) {
                    item.IconPath = mlist.Where(x => x.Id == item.Icon).FirstOrDefault()?.Path ?? "";
                    item.BigIconPath = mlist.Where(x => x.Id == item.BigIcon).FirstOrDefault()?.Path ?? "";
                    item.AdvertisingImagePath = mlist.Where(x => x.Id == item.AdvertisingImage).FirstOrDefault()?.Path ?? "";
                }
            }
            return list;
        }

        /// <summary>
        /// 删除分类
        /// </summary>
        /// <param name="categoryId"></param>
        /// <param name="tenantId"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool DelProductCategoryInfo(int categoryId, int tenantId, int mallBaseId)
        {
            var trans = product_CategoryRepository.DbTransaction;
            try
            {
                var list = GetProductCategoryForALLChlid(categoryId, tenantId, mallBaseId);
                foreach (var item in list) {
                    Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                        { nameof(RB_Product_Category.Status),1},
                        { nameof(RB_Product_Category.UpdateDate),DateTime.Now},
                    };
                    List<WhereHelper> wheres = new List<WhereHelper>() {
                        new WhereHelper(){
                             FiledName=nameof(RB_Product_Category.Id),
                              FiledValue=categoryId,
                               OperatorEnum=OperatorEnum.Equal
                        },
                        new WhereHelper(){
                             FiledName=nameof(RB_Product_Category.TenantId),
                              FiledValue=tenantId,
                               OperatorEnum=OperatorEnum.Equal
                        },
                        new WhereHelper(){
                             FiledName=nameof(RB_Product_Category.MallBaseId),
                              FiledValue=mallBaseId,
                               OperatorEnum=OperatorEnum.Equal
                        }
                    };
                    bool flag = product_CategoryRepository.Update(keyValues, wheres, trans);
                }
                product_CategoryRepository.DBSession.Commit();
                return true;
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, "DelProductCategoryInfo");
                product_CategoryRepository.DBSession.Rollback();
                return false;
            }
        }

        /// <summary>
        /// 获取分类样式
        /// </summary>
        /// <param name="tenantId"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public RB_Product_CategoryStyle_Extend GetProductCategoryStyleInfo(int tenantId, int mallBaseId)
        {
            return product_CategoryStyleRepository.GetList(new RB_Product_CategoryStyle_Extend() { TenantId = tenantId, MallBaseId = mallBaseId }).FirstOrDefault();
        }

        /// <summary>
        /// 保存分类样式
        /// </summary>
        /// <param name="demodel"></param>
        /// <returns></returns>
        public bool SetProductCategoryStyleInfo(RB_Product_CategoryStyle_Extend demodel)
        {
            if (demodel.Id > 0)
            {
                Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                    { nameof(RB_Product_CategoryStyle.Type),demodel.Type},
                    { nameof(RB_Product_CategoryStyle.Style),demodel.Style},
                    { nameof(RB_Product_CategoryStyle.CategoryNum),demodel.CategoryNum},
                    { nameof(RB_Product_CategoryStyle.RowNum),demodel.RowNum},
                    { nameof(RB_Product_CategoryStyle.UpdateDate),demodel.UpdateDate},
                };
                List<WhereHelper> wheres = new List<WhereHelper>() {
                    new WhereHelper(){
                            FiledName=nameof(RB_Product_Category.Id),
                            FiledValue=demodel.Id,
                            OperatorEnum=OperatorEnum.Equal
                    }
                };
                return product_CategoryStyleRepository.Update(keyValues, wheres);
            }
            else {
                return product_CategoryStyleRepository.Insert(demodel) > 0;
            }
        }

        /// <summary>
        /// 保存排序
        /// </summary>
        /// <param name="delist"></param>
        /// <param name="tenantId"></param>
        /// <param name="mallBaseId"></param>
        /// <returns></returns>
        public bool SetProductCategorySort(List<RB_Product_CategorySort> delist, int tenantId, int mallBaseId)
        {
            var trans = product_CategoryRepository.DbTransaction;
            try
            {
                string ids = string.Join(",", delist.Select(x => x.Id));
                var list = product_CategoryRepository.GetList(new RB_Product_Category_Extend() { CategoryIds = ids, TenantId = tenantId, MallBaseId = mallBaseId });
                if (list.Any())
                {
                    foreach (var item in list)
                    {
                        var demodel = delist.Where(x => x.Id == item.Id).FirstOrDefault();
                        Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                            { nameof(RB_Product_Category.Sort),demodel.Sort},
                        };
                        List<WhereHelper> wheres = new List<WhereHelper>() {
                            new WhereHelper(){
                                 FiledName=nameof(RB_Product_Category.Id),
                                  FiledValue=item.Id,
                                   OperatorEnum=OperatorEnum.Equal
                            }
                        };
                        bool flag = product_CategoryRepository.Update(keyValues, wheres, trans);
                    }
                }
                product_CategoryRepository.DBSession.Commit();
                return true;
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, "SetProductCategorySort");
                product_CategoryRepository.DBSession.Rollback();
                return false;
            }
        }


        /// <summary>
        /// 获取树形结构列表
        /// </summary>
        /// <param name="demodel"></param>
        /// <returns></returns>
        public List<RB_Product_CategoryTree> GetProductCategoryTreeList(RB_Product_Category_Extend demodel)
        {
            List<RB_Product_CategoryTree> treeList = new List<RB_Product_CategoryTree>();
            var list = product_CategoryRepository.GetList(demodel);
            if (list.Any())
            {
                List<int> ImgIds = new List<int>();
                ImgIds.AddRange(list.Select(x => x.Icon ?? 0).Distinct());
                ImgIds.AddRange(list.Select(x => x.BigIcon ?? 0).Distinct());
                ImgIds.AddRange(list.Select(x => x.AdvertisingImage ?? 0).Distinct());
                List<RB_Material_Info_Extend> mlist = new List<RB_Material_Info_Extend>();
                if (ImgIds.Any())
                {
                    mlist = material_InfoRepository.GetList(new RB_Material_Info_Extend() { MaterialIds = string.Join(",", ImgIds), TenantId = demodel.TenantId, MallBaseId = demodel.MallBaseId });
                }

                foreach (var item in list)
                {
                    item.IconPath = mlist.Where(x => x.Id == item.Icon).FirstOrDefault()?.Path ?? "";
                    item.BigIconPath = mlist.Where(x => x.Id == item.BigIcon).FirstOrDefault()?.Path ?? "";
                    item.AdvertisingImagePath = mlist.Where(x => x.Id == item.AdvertisingImage).FirstOrDefault()?.Path ?? "";
                }
                list = list.OrderBy(x => x.Sort).ThenBy(x => x.Id).ToList();
                treeList = GetProductCategoryChildTree(list, 0);                
            }

            return treeList;
        }

        /// <summary>
        /// 获取子集列表
        /// </summary>
        /// <param name="list"></param>
        /// <param name="parentId"></param>
        /// <returns></returns>
        public List<RB_Product_CategoryTree> GetProductCategoryChildTree(List<RB_Product_Category_Extend> list, int parentId) {
            List<RB_Product_CategoryTree> RList = new List<RB_Product_CategoryTree>();
            var CList = list.Where(x => x.ParentId == parentId).ToList();
            foreach (var item in CList) {
                var cmodel = new RB_Product_CategoryTree
                {
                    Id = item.Id,
                    Name = item.Name,
                    Icon = item.Icon ?? 0,
                    IconPath = item.IconPath
                };
                cmodel.ChildList = GetProductCategoryChildTree(list, item.Id);
                RList.Add(cmodel);
            }
            return RList;
        }

        /// <summary>
        /// 保存商品分类
        /// </summary>
        /// <param name="demodel"></param>
        /// <returns></returns>
        public bool SetProductCategoryInfo(RB_Product_Category_Extend demodel)
        {
            if (demodel.Id > 0)
            {
                Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                    { nameof(RB_Product_Category.AdvertisingAddress),demodel.AdvertisingAddress},
                    { nameof(RB_Product_Category.AdvertisingImage),demodel.AdvertisingImage},
                    { nameof(RB_Product_Category.BigIcon),demodel.BigIcon},
                    { nameof(RB_Product_Category.Enabled),demodel.Enabled},
                    { nameof(RB_Product_Category.Icon),demodel.Icon},
                    { nameof(RB_Product_Category.IsShow),demodel.IsShow},
                    { nameof(RB_Product_Category.Name),demodel.Name},
                    { nameof(RB_Product_Category.Sort),demodel.Sort},
                    { nameof(RB_Product_Category.UpdateDate),demodel.UpdateDate}
                };
                List<WhereHelper> wheres = new List<WhereHelper>() {
                    new WhereHelper(){
                         FiledName=nameof(RB_Product_Category.Id),
                          FiledValue=demodel.Id,
                           OperatorEnum=OperatorEnum.Equal
                    }
                };
                return product_CategoryRepository.Update(keyValues, wheres);
            }
            else
            {
                int Id = product_CategoryRepository.Insert(demodel);
                if (Id > 0) {
                    int RootId = Id;
                    if (demodel.ParentId > 0) {
                        var pmodel = product_CategoryRepository.GetEntity(demodel.ParentId);
                        RootId = pmodel?.RootId ?? 0;
                    }
                    Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                        { nameof(RB_Product_Category.RootId),RootId},
                    };
                    List<WhereHelper> wheres = new List<WhereHelper>() {
                        new WhereHelper(){
                                FiledName=nameof(RB_Product_Category.Id),
                                FiledValue=Id,
                                OperatorEnum=OperatorEnum.Equal
                        }
                    };
                    product_CategoryRepository.Update(keyValues, wheres);
                }
                return Id > 0;
            }
        }

        /// <summary>
        /// 获取当前分类的所有子集分类
        /// </summary>
        /// <param name="categoryId"></param>
        /// <param name="tid"></param>
        /// <param name="mid"></param>
        /// <returns></returns>
        public List<RB_Product_Category_Extend> GetProductCategoryForALLChlid(int categoryId,int tid,int mid)
        {
            var model = product_CategoryRepository.GetEntity(categoryId);
            var list = product_CategoryRepository.GetList(new RB_Product_Category_Extend() { RootId = model.RootId, TenantId = tid, MallBaseId = mid });
            List<RB_Product_Category_Extend> RList = GetProductCategoryALLChild(list, model.Id);
            RList.AddRange(list.Where(x => x.Id == model.Id).ToList());
            return RList;
        }

        /// <summary>
        /// 获取所有子集
        /// </summary>
        /// <param name="list"></param>
        /// <param name="id"></param>
        /// <returns></returns>
        private List<RB_Product_Category_Extend> GetProductCategoryALLChild(List<RB_Product_Category_Extend> list, int id)
        {
            List<RB_Product_Category_Extend> RList = new List<RB_Product_Category_Extend>();
            var list2 = list.Where(x => x.ParentId == id).ToList();
            RList.AddRange(list2);
            foreach (var item in list2) {
                RList.AddRange(GetProductCategoryALLChild(list, item.Id));
            }
            return RList;
        }

        /// <summary>
        /// 分类转移
        /// </summary>
        /// <param name="cmodel"></param>
        /// <param name="parentId"></param>
        /// <param name="list"></param>
        /// <returns></returns>
        public bool SetProductCategoryTransfer(RB_Product_Category_Extend cmodel, int parentId, List<RB_Product_Category_Extend> list)
        {
            var trans = product_CategoryRepository.DbTransaction;
            try
            {
                int RootId = cmodel.Id;
                int Tier = 0;
                if (parentId > 0)
                {
                    var pmodel = product_CategoryRepository.GetEntity(parentId);
                    if (pmodel == null)
                    {
                        return false;
                    }
                    RootId = pmodel.RootId ?? 0;
                    Tier = (pmodel.Tier ?? 0) + 1;
                }
                else
                {
                    Tier = 1;
                }
                Dictionary<string, object> keyValues = new Dictionary<string, object>() {
                    { nameof(RB_Product_Category.ParentId),parentId},
                    { nameof(RB_Product_Category.UpdateDate),DateTime.Now},
                };
                List<WhereHelper> wheres = new List<WhereHelper>() {
                    new WhereHelper(){
                         FiledName=nameof(RB_Product_Category.Id),
                          FiledValue=cmodel.Id,
                           OperatorEnum=OperatorEnum.Equal
                    }
                };
                bool flag = product_CategoryRepository.Update(keyValues, wheres, trans);
                if (flag)
                {
                    //更新所有子集rootid=
                    foreach (var item in list)
                    {
                        Dictionary<string, object> keyValues1 = new Dictionary<string, object>() {
                            { nameof(RB_Product_Category.RootId),RootId},
                            { nameof(RB_Product_Category.Tier),Tier+(item.Tier??0)-(cmodel.Tier??0) }
                        };
                        List<WhereHelper> wheres1 = new List<WhereHelper>() {
                            new WhereHelper(){
                                    FiledName=nameof(RB_Product_Category.Id),
                                    FiledValue=item.Id,
                                    OperatorEnum=OperatorEnum.Equal
                            }
                        };
                        product_CategoryRepository.Update(keyValues1, wheres1, trans);
                    }
                }
                product_CategoryRepository.DBSession.Commit();
                return flag;
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, "SetProductCategoryTransfer");
                product_CategoryRepository.DBSession.Rollback();
                return false;
            }
        }

        #endregion
    }
}
