﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Linq.Expressions;
using System.Collections.ObjectModel;

namespace DapperExtensions.Lambda
{
    /// <summary>
    /// ExpressionVisitor
    /// </summary>
    public abstract class ExpressionVisitor
    {
        /// <summary>
        /// 构造函数
        /// </summary>
        protected ExpressionVisitor() { }

        /// <summary>
        /// Visit
        /// </summary>
        /// <param name="exp"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression Visit(System.Linq.Expressions.Expression exp)
        {
            if (exp == null)
                return exp;
            switch (exp.NodeType)
            {
                case ExpressionType.Negate:
                case ExpressionType.NegateChecked:
                case ExpressionType.Not:
                case ExpressionType.Convert:
                case ExpressionType.ConvertChecked:
                case ExpressionType.ArrayLength:
                case ExpressionType.Quote:
                case ExpressionType.TypeAs:
                    return this.VisitUnary((UnaryExpression)exp);
                case ExpressionType.Add:
                case ExpressionType.AddChecked:
                case ExpressionType.Subtract:
                case ExpressionType.SubtractChecked:
                case ExpressionType.Multiply:
                case ExpressionType.MultiplyChecked:
                case ExpressionType.Divide:
                case ExpressionType.Modulo:
                case ExpressionType.And:
                case ExpressionType.AndAlso:
                case ExpressionType.Or:
                case ExpressionType.OrElse:
                case ExpressionType.LessThan:
                case ExpressionType.LessThanOrEqual:
                case ExpressionType.GreaterThan:
                case ExpressionType.GreaterThanOrEqual:
                case ExpressionType.Equal:
                case ExpressionType.NotEqual:
                case ExpressionType.Coalesce:
                case ExpressionType.ArrayIndex:
                case ExpressionType.RightShift:
                case ExpressionType.LeftShift:
                case ExpressionType.ExclusiveOr:
                    return this.VisitBinary((BinaryExpression)exp);
                case ExpressionType.TypeIs:
                    return this.VisitTypeIs((TypeBinaryExpression)exp);
                case ExpressionType.Conditional:
                    return this.VisitConditional((ConditionalExpression)exp);
                case ExpressionType.Constant:
                    return this.VisitConstant((ConstantExpression)exp);
                case ExpressionType.Parameter:
                    return this.VisitParameter((ParameterExpression)exp);
                case ExpressionType.MemberAccess:
                    return this.VisitMemberAccess((MemberExpression)exp);
                case ExpressionType.Call:
                    return this.VisitMethodCall((MethodCallExpression)exp);
                case ExpressionType.Lambda:
                    return this.VisitLambda((LambdaExpression)exp);
                case ExpressionType.New:
                    return this.VisitNew((NewExpression)exp);
                case ExpressionType.NewArrayInit:
                case ExpressionType.NewArrayBounds:
                    return this.VisitNewArray((NewArrayExpression)exp);
                case ExpressionType.Invoke:
                    return this.VisitInvocation((InvocationExpression)exp);
                case ExpressionType.MemberInit:
                    return this.VisitMemberInit((MemberInitExpression)exp);
                case ExpressionType.ListInit:
                    return this.VisitListInit((ListInitExpression)exp);
                default:
                    throw new Exception(string.Format("Unhandled expression type: '{0}'", exp.NodeType));
            }
        }

        /// <summary>
        /// VisitBinding
        /// </summary>
        /// <param name="binding"></param>
        /// <returns></returns>
        protected virtual MemberBinding VisitBinding(MemberBinding binding)
        {
            switch (binding.BindingType)
            {
                case MemberBindingType.Assignment:
                    return this.VisitMemberAssignment((MemberAssignment)binding);
                case MemberBindingType.MemberBinding:
                    return this.VisitMemberMemberBinding((MemberMemberBinding)binding);
                case MemberBindingType.ListBinding:
                    return this.VisitMemberListBinding((MemberListBinding)binding);
                default:
                    throw new Exception(string.Format("Unhandled binding type '{0}'", binding.BindingType));
            }
        }

        /// <summary>
        /// VisitElementInitializer
        /// </summary>
        /// <param name="initializer"></param>
        /// <returns></returns>
        protected virtual ElementInit VisitElementInitializer(ElementInit initializer)
        {
            ReadOnlyCollection<System.Linq.Expressions.Expression> arguments = this.VisitExpressionList(initializer.Arguments);
            if (arguments != initializer.Arguments)
            {
                return System.Linq.Expressions.Expression.ElementInit(initializer.AddMethod, arguments);
            }
            return initializer;
        }

        /// <summary>
        /// VisitUnary
        /// </summary>
        /// <param name="u"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitUnary(UnaryExpression u)
        {
            System.Linq.Expressions.Expression operand = this.Visit(u.Operand);
            if (operand != u.Operand)
            {
                return System.Linq.Expressions.Expression.MakeUnary(u.NodeType, operand, u.Type, u.Method);
            }
            return u;
        }

        /// <summary>
        /// VisitBinary
        /// </summary>
        /// <param name="b"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitBinary(BinaryExpression b)
        {
            System.Linq.Expressions.Expression left = this.Visit(b.Left);
            System.Linq.Expressions.Expression right = this.Visit(b.Right);
            System.Linq.Expressions.Expression conversion = this.Visit(b.Conversion);
            if (left != b.Left || right != b.Right || conversion != b.Conversion)
            {
                if (b.NodeType == ExpressionType.Coalesce && b.Conversion != null)
                    return System.Linq.Expressions.Expression.Coalesce(left, right, conversion as LambdaExpression);
                else
                    return System.Linq.Expressions.Expression.MakeBinary(b.NodeType, left, right, b.IsLiftedToNull, b.Method);
            }
            return b;
        }

        /// <summary>
        /// VisitTypeIs
        /// </summary>
        /// <param name="b"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitTypeIs(TypeBinaryExpression b)
        {
            System.Linq.Expressions.Expression expr = this.Visit(b.Expression);
            if (expr != b.Expression)
            {
                return System.Linq.Expressions.Expression.TypeIs(expr, b.TypeOperand);
            }
            return b;
        }

        /// <summary>
        /// VisitConstant
        /// </summary>
        /// <param name="c"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitConstant(ConstantExpression c)
        {
            return c;
        }

        /// <summary>
        /// VisitConditional
        /// </summary>
        /// <param name="c"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitConditional(ConditionalExpression c)
        {
            System.Linq.Expressions.Expression test = this.Visit(c.Test);
            System.Linq.Expressions.Expression ifTrue = this.Visit(c.IfTrue);
            System.Linq.Expressions.Expression ifFalse = this.Visit(c.IfFalse);
            if (test != c.Test || ifTrue != c.IfTrue || ifFalse != c.IfFalse)
            {
                return System.Linq.Expressions.Expression.Condition(test, ifTrue, ifFalse);
            }
            return c;
        }

        /// <summary>
        /// VisitParameter
        /// </summary>
        /// <param name="p"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitParameter(ParameterExpression p)
        {
            return p;
        }

        /// <summary>
        /// VisitMemberAccess
        /// </summary>
        /// <param name="m"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitMemberAccess(MemberExpression m)
        {
            System.Linq.Expressions.Expression exp = this.Visit(m.Expression);
            if (exp != m.Expression)
            {
                return System.Linq.Expressions.Expression.MakeMemberAccess(exp, m.Member);
            }
            return m;
        }

        /// <summary>
        /// VisitMethodCall
        /// </summary>
        /// <param name="m"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitMethodCall(MethodCallExpression m)
        {
            System.Linq.Expressions.Expression obj = this.Visit(m.Object);
            IEnumerable<System.Linq.Expressions.Expression> args = this.VisitExpressionList(m.Arguments);
            if (obj != m.Object || args != m.Arguments)
            {
                return System.Linq.Expressions.Expression.Call(obj, m.Method, args);
            }
            return m;
        }

        /// <summary>
        /// VisitExpressionList
        /// </summary>
        /// <param name="original"></param>
        /// <returns></returns>
        protected virtual ReadOnlyCollection<System.Linq.Expressions.Expression> VisitExpressionList(ReadOnlyCollection<System.Linq.Expressions.Expression> original)
        {
            List<System.Linq.Expressions.Expression> list = null;
            for (int i = 0, n = original.Count; i < n; i++)
            {
                System.Linq.Expressions.Expression p = this.Visit(original[i]);
                if (list != null)
                {
                    list.Add(p);
                }
                else if (p != original[i])
                {
                    list = new List<System.Linq.Expressions.Expression>(n);
                    for (int j = 0; j < i; j++)
                    {
                        list.Add(original[j]);
                    }
                    list.Add(p);
                }
            }
            if (list != null)
            {
                return list.AsReadOnly();
            }
            return original;
        }


        /// <summary>
        /// VisitMemberAssignment
        /// </summary>
        /// <param name="assignment"></param>
        /// <returns></returns>
        protected virtual MemberAssignment VisitMemberAssignment(MemberAssignment assignment)
        {
            System.Linq.Expressions.Expression e = this.Visit(assignment.Expression);
            if (e != assignment.Expression)
            {
                return System.Linq.Expressions.Expression.Bind(assignment.Member, e);
            }
            return assignment;
        }

        /// <summary>
        /// VisitMemberMemberBinding
        /// </summary>
        /// <param name="binding"></param>
        /// <returns></returns>
        protected virtual MemberMemberBinding VisitMemberMemberBinding(MemberMemberBinding binding)
        {
            IEnumerable<MemberBinding> bindings = this.VisitBindingList(binding.Bindings);
            if (bindings != binding.Bindings)
            {
                return System.Linq.Expressions.Expression.MemberBind(binding.Member, bindings);
            }
            return binding;
        }

        /// <summary>
        /// VisitMemberListBinding
        /// </summary>
        /// <param name="binding"></param>
        /// <returns></returns>

        protected virtual MemberListBinding VisitMemberListBinding(MemberListBinding binding)
        {
            IEnumerable<ElementInit> initializers = this.VisitElementInitializerList(binding.Initializers);
            if (initializers != binding.Initializers)
            {
                return System.Linq.Expressions.Expression.ListBind(binding.Member, initializers);
            }
            return binding;
        }

        /// <summary>
        /// VisitBindingList
        /// </summary>
        /// <param name="original"></param>
        /// <returns></returns>
        protected virtual IEnumerable<MemberBinding> VisitBindingList(ReadOnlyCollection<MemberBinding> original)
        {
            List<MemberBinding> list = null;
            for (int i = 0, n = original.Count; i < n; i++)
            {
                MemberBinding b = this.VisitBinding(original[i]);
                if (list != null)
                {
                    list.Add(b);
                }
                else if (b != original[i])
                {
                    list = new List<MemberBinding>(n);
                    for (int j = 0; j < i; j++)
                    {
                        list.Add(original[j]);
                    }
                    list.Add(b);
                }
            }
            if (list != null)
                return list;
            return original;
        }

        /// <summary>
        /// VisitElementInitializerList
        /// </summary>
        /// <param name="original"></param>
        /// <returns></returns>
        protected virtual IEnumerable<ElementInit> VisitElementInitializerList(ReadOnlyCollection<ElementInit> original)
        {
            List<ElementInit> list = null;
            for (int i = 0, n = original.Count; i < n; i++)
            {
                ElementInit init = this.VisitElementInitializer(original[i]);
                if (list != null)
                {
                    list.Add(init);
                }
                else if (init != original[i])
                {
                    list = new List<ElementInit>(n);
                    for (int j = 0; j < i; j++)
                    {
                        list.Add(original[j]);
                    }
                    list.Add(init);
                }
            }
            if (list != null)
                return list;
            return original;
        }

        /// <summary>
        /// VisitLambda
        /// </summary>
        /// <param name="lambda"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitLambda(LambdaExpression lambda)
        {
            System.Linq.Expressions.Expression body = this.Visit(lambda.Body);
            if (body != lambda.Body)
            {
                return System.Linq.Expressions.Expression.Lambda(lambda.Type, body, lambda.Parameters);
            }
            return lambda;
        }

        /// <summary>
        /// VisitNew
        /// </summary>
        /// <param name="nex"></param>
        /// <returns></returns>
        protected virtual NewExpression VisitNew(NewExpression nex)
        {
            IEnumerable<System.Linq.Expressions.Expression> args = this.VisitExpressionList(nex.Arguments);
            if (args != nex.Arguments)
            {
                if (nex.Members != null)
                    return System.Linq.Expressions.Expression.New(nex.Constructor, args, nex.Members);
                else
                    return System.Linq.Expressions.Expression.New(nex.Constructor, args);
            }
            return nex;
        }

        /// <summary>
        /// VisitMemberInit
        /// </summary>
        /// <param name="init"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitMemberInit(MemberInitExpression init)
        {
            NewExpression n = this.VisitNew(init.NewExpression);
            IEnumerable<MemberBinding> bindings = this.VisitBindingList(init.Bindings);
            if (n != init.NewExpression || bindings != init.Bindings)
            {
                return System.Linq.Expressions.Expression.MemberInit(n, bindings);
            }
            return init;
        }


        /// <summary>
        /// VisitListInit
        /// </summary>
        /// <param name="init"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitListInit(ListInitExpression init)
        {
            NewExpression n = this.VisitNew(init.NewExpression);
            IEnumerable<ElementInit> initializers = this.VisitElementInitializerList(init.Initializers);
            if (n != init.NewExpression || initializers != init.Initializers)
            {
                return System.Linq.Expressions.Expression.ListInit(n, initializers);
            }
            return init;
        }

        /// <summary>
        /// VisitNewArray
        /// </summary>
        /// <param name="na"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitNewArray(NewArrayExpression na)
        {
            IEnumerable<System.Linq.Expressions.Expression> exprs = this.VisitExpressionList(na.Expressions);
            if (exprs != na.Expressions)
            {
                if (na.NodeType == ExpressionType.NewArrayInit)
                {
                    return System.Linq.Expressions.Expression.NewArrayInit(na.Type.GetElementType(), exprs);
                }
                else
                {
                    return System.Linq.Expressions.Expression.NewArrayBounds(na.Type.GetElementType(), exprs);
                }
            }
            return na;
        }

        /// <summary>
        /// VisitInvocation
        /// </summary>
        /// <param name="iv"></param>
        /// <returns></returns>
        protected virtual System.Linq.Expressions.Expression VisitInvocation(InvocationExpression iv)
        {
            IEnumerable<System.Linq.Expressions.Expression> args = this.VisitExpressionList(iv.Arguments);
            System.Linq.Expressions.Expression expr = this.Visit(iv.Expression);
            if (args != iv.Arguments || expr != iv.Expression)
            {
                return System.Linq.Expressions.Expression.Invoke(expr, args);
            }
            return iv;
        }
    }
}
