﻿using System;
using System.Collections.Generic;
using System.Text;

namespace DapperExtensions.Sql
{
    /// <summary>
    /// SqlCe语句配置扩展类
    /// </summary>
    public class SqlCeDialect : SqlDialectBase
    {
        /// <summary>
        /// 开始字符
        /// </summary>
        public override char OpenQuote
        {
            get { return '['; }
        }

        /// <summary>
        /// 结束字符
        /// </summary>
        public override char CloseQuote
        {
            get { return ']'; }
        }

        /// <summary>
        ///  是否支持多项结构
        /// </summary>
        public override bool SupportsMultipleStatements
        {
            get { return false; }
        }

        /// <summary>
        /// 获取表名
        /// </summary>
        /// <param name="schemaName">架构名(.dbo)</param>
        /// <param name="tableName">表名</param>
        /// <param name="alias">别名</param>
        /// <param name="dbName">数据库名称</param>
        /// <returns></returns>
        public override string GetTableName(string schemaName, string tableName, string alias, string dbName = "")
        {
            if (string.IsNullOrWhiteSpace(tableName))
            {
                throw new ArgumentNullException("TableName");
            }

            StringBuilder result = new StringBuilder();
            result.Append(OpenQuote);
            if (!string.IsNullOrWhiteSpace(schemaName))
            {
                result.AppendFormat("{0}_", schemaName);
            }
            result.AppendFormat("{0}{1}", tableName, CloseQuote);
            if (!string.IsNullOrWhiteSpace(alias))
            {
                result.AppendFormat(" AS {0}{1}{2}", OpenQuote, alias, CloseQuote);
            }
            return result.ToString();
        }

        /// <summary>
        /// 获取自增Sql
        /// </summary>
        /// <param name="tableName">表名</param>
        /// <returns></returns>
        public override string GetIdentitySql(string tableName)
        {
            return "SELECT CAST(@@IDENTITY AS BIGINT) AS [Id]";
        }

        /// <summary>
        /// 获取分页的Sql
        /// </summary>
        /// <param name="sql">sql命令</param>
        /// <param name="page">页码</param>
        /// <param name="resultsPerPage"></param>
        /// <param name="parameters">参数字典</param>
        /// <returns></returns>
        public override string GetPagingSql(string sql, int page, int resultsPerPage, IDictionary<string, object> parameters)
        {
            int startValue = (page * resultsPerPage);
            return GetSetSql(sql, startValue, resultsPerPage, parameters);
        }

        /// <summary>
        /// 设置Sql
        /// </summary>
        /// <param name="sql">sql命令</param>
        /// <param name="firstResult">第一结果</param>
        /// <param name="maxResults">最大结果</param>
        /// <param name="parameters">参数</param>
        /// <returns></returns>
        public override string GetSetSql(string sql, int firstResult, int maxResults, IDictionary<string, object> parameters)
        {
            string result = string.Format("{0} OFFSET @firstResult ROWS FETCH NEXT @maxResults ROWS ONLY", sql);
            parameters.Add("@firstResult", firstResult);
            parameters.Add("@maxResults", maxResults);
            return result;
        }
    }
}