﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Mall.Common.API;
using Mall.Model.Extend.Product;
using Mall.Module.User;
using Mall.WebApi.Filter;
using Microsoft.AspNetCore.Cors;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Mall.Common.Plugin;
using Mall.Common.Enum.User;
using Mall.CacheManager.User;
using Newtonsoft.Json.Linq;
using Mall.Common;
using Mall.Module.Product;

namespace Mall.WebApi.Controllers.MallBase
{
    [Route("api/[controller]/[action]")]
    [ApiExceptionFilter]
    [ApiController]
    [EnableCors("AllowCors")]
    public class ProductController : BaseController
    {

        private readonly ProductModule productModule = new ProductModule();

        #region 素材管理
        /// <summary>
        /// 获取素材分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetMaterialInfoPageList()
        {
            var parms = RequestParm;
            ResultPageModel pagelist = JsonConvert.DeserializeObject<ResultPageModel>(parms.msg.ToString());
            RB_Material_Info_Extend demodel = JsonConvert.DeserializeObject<RB_Material_Info_Extend>(parms.msg.ToString());

            demodel.TenantId = Convert.ToInt32(parms.uid);
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetMaterialInfoPageList(pagelist.pageIndex, pagelist.pageSize, out long count, demodel);
            pagelist.count = Convert.ToInt32(count);
            pagelist.pageData = list.Select(x => new
            {
                x.Id,
                x.Name,
                x.Type,
                x.GroupId,
                x.Path,
                x.Image,
                x.VideoTime,
                x.WithHeight,
                x.Recycled,
                x.TenantId,
                x.MallBaseId,
                CreateDate = x.CreateDate.HasValue ? x.CreateDate.Value.ToString("yyyy-MM-dd") : ""
            });
            return ApiResult.Success("", pagelist);
        }

        /// <summary>
        /// 新增素材信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialInfo() {
            var parms = RequestParm;
            RB_Material_Info_Extend demodel = JsonConvert.DeserializeObject<RB_Material_Info_Extend>(parms.msg.ToString());
            if (string.IsNullOrEmpty(demodel.Name))
            {
                return ApiResult.ParamIsNull("请输入分组名称");
            }
            if (demodel.Type != 1 && demodel.Type != 2)
            {
                return ApiResult.ParamIsNull("请传递类型");
            }
            if ((demodel.GroupId ?? 0) < 0) {
                return ApiResult.ParamIsNull("请输入分组");
            }
            if (string.IsNullOrEmpty(demodel.Path)) {
                return ApiResult.ParamIsNull("请传递文件路径");
            }

            demodel.Path = Config.GetFilePath(demodel.Path);
            demodel.Image = Config.GetFilePath(demodel.Image);
            demodel.TenantId = Convert.ToInt32(parms.uid);
            demodel.MallBaseId = parms.MallBaseId;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            demodel.Status = 0;
            demodel.Recycled = 1;
            bool flag = productModule.SetMaterialInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 获取素材分组列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetMaterialGroupList()
        {
            var parms = RequestParm;
            RB_Material_Group_Extend demodel = JsonConvert.DeserializeObject<RB_Material_Group_Extend>(parms.msg.ToString());

            demodel.TenantId = Convert.ToInt32(parms.uid);
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetMaterialGroupList(demodel);
            return ApiResult.Success("", list.Select(x => new
            {
                x.Id,
                x.Name,
                x.Type,
                x.Sort,
                x.Recycled,
                x.TenantId,
                x.MallBaseId,
                CreateDate = x.CreateDate.HasValue ? x.CreateDate.Value.ToString("yyyy-MM-dd") : ""
            }));
        }

        /// <summary>
        /// 新增修改分组
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialGroupInfo() {
            var parms = RequestParm;
            RB_Material_Group_Extend demodel = JsonConvert.DeserializeObject<RB_Material_Group_Extend>(parms.msg.ToString());
            if (string.IsNullOrEmpty(demodel.Name)) {
                return ApiResult.ParamIsNull("请输入分组名称");
            }
            if (demodel.Type != 1 && demodel.Type != 2) {
                return ApiResult.ParamIsNull("请传递类型");
            }

            demodel.TenantId = Convert.ToInt32(parms.uid);
            demodel.MallBaseId = parms.MallBaseId;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            demodel.Status = 0;
            demodel.Recycled = 1;
            bool flag = productModule.SetMaterialGroupInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 设置分组回收
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialGroupRecycled() {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            int GroupId = parms.GetInt("GroupId", 0);
            if (GroupId < 0) {
                return ApiResult.ParamIsNull("请传递分组id");
            }

            bool flag = productModule.SetMaterialGroupRecycled(GroupId, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 删除分组信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelMaterialGroupInfo()
        {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            int GroupId = parms.GetInt("GroupId", 0);
            if (GroupId < 0)
            {
                return ApiResult.ParamIsNull("请传递分组id");
            }

            bool flag = productModule.DelMaterialGroupInfo(GroupId, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 设置分组还原
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialGroupRestore()
        {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            int GroupId = parms.GetInt("GroupId", 0);
            if (GroupId < 0)
            {
                return ApiResult.ParamIsNull("请传递分组id");
            }

            bool flag = productModule.SetMaterialGroupRestore(GroupId, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }


        /// <summary>
        /// 设置素材回收
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialInfoRecycled()
        {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            string MaterialIds = parms.GetStringValue("MaterialIds");
            if (string.IsNullOrEmpty(MaterialIds))
            {
                return ApiResult.ParamIsNull("请传递素材ids");
            }

            bool flag = productModule.SetMaterialInfoRecycled(MaterialIds, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 删除素材信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelMaterialInfo()
        {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            string MaterialIds = parms.GetStringValue("MaterialIds");
            if (string.IsNullOrEmpty(MaterialIds))
            {
                return ApiResult.ParamIsNull("请传递素材ids");
            }

            bool flag = productModule.DelMaterialInfo(MaterialIds, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 设置素材还原
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialInfoRestore()
        {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            string MaterialIds = parms.GetStringValue("MaterialIds");
            if (string.IsNullOrEmpty(MaterialIds))
            {
                return ApiResult.ParamIsNull("请传递素材ids");
            }

            bool flag = productModule.SetMaterialInfoRestore(MaterialIds, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 设置素材转移
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetMaterialInfoTransfer()
        {
            var requestParm = RequestParm;
            var parms = JObject.Parse(requestParm.msg.ToString());
            string MaterialIds = parms.GetStringValue("MaterialIds");
            int GroupId = parms.GetInt("GroupId", 0);
            int Type = parms.GetInt("Type", 0);
            if (string.IsNullOrEmpty(MaterialIds))
            {
                return ApiResult.ParamIsNull("请传递素材ids");
            }
            if (GroupId <= 0) 
            {
                return ApiResult.ParamIsNull("请传递分组id");
            }

            bool flag = productModule.SetMaterialInfoTransfer(MaterialIds, GroupId, Type, requestParm.uid, requestParm.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        #endregion

        #region 商品分类

        /// <summary>
        /// 获取商品分类树形列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductCategoryPageList()
        {
            var parms = RequestParm;
            ResultPageModel pagelist = JsonConvert.DeserializeObject<ResultPageModel>(parms.msg.ToString());
            RB_Product_Category_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Category_Extend>(parms.msg.ToString());

            demodel.TenantId = Convert.ToInt32(parms.uid);
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProductCategoryPageList(pagelist.pageIndex, pagelist.pageSize, out long count, demodel);
            pagelist.count = Convert.ToInt32(count);
            pagelist.pageData = list.Select(x => new
            {
                x.Id,
                x.Name,
                x.Tier,
                x.ParentId,
                x.Sort,
                x.Icon,
                x.IconPath,
                x.BigIcon,
                x.BigIconPath,
                x.AdvertisingImage,
                x.AdvertisingImagePath,
                x.AdvertisingAddress,
                x.Enabled,
                x.IsShow,
                x.TenantId,
                x.MallBaseId,
                CreateDate = x.CreateDate.HasValue ? x.CreateDate.Value.ToString("yyyy-MM-dd") : ""
            });
            return ApiResult.Success("", pagelist);
        }

        /// <summary>
        /// 获取分类树形结构
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductCategoryTreeList() {
            var parms = RequestParm;
            RB_Product_Category_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Category_Extend>(parms.msg.ToString());

            demodel.TenantId = Convert.ToInt32(parms.uid);
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProductCategoryTreeList(demodel);
            return ApiResult.Success("", list);
        }

        /// <summary>
        /// 获取分类详情
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductCategoryInfo() {
            var req = RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int CategoryId = parms.GetInt("CategoryId", 0);
            if (CategoryId <= 0) {
                return ApiResult.ParamIsNull();
            }

            var model = productModule.GetProductCategoryInfo(CategoryId);
            return ApiResult.Success("", new
            {
                model.Id,
                model.Name,
                model.Tier,
                model.ParentId,
                model.Sort,
                model.Icon,
                model.IconPath,
                model.BigIcon,
                model.BigIconPath,
                model.AdvertisingImage,
                model.AdvertisingImagePath,
                model.AdvertisingAddress,
                model.Enabled,
                model.IsShow,
                model.TenantId,
                model.MallBaseId,
                CreateDate = model.CreateDate.HasValue ? model.CreateDate.Value.ToString("yyyy-MM-dd") : ""
            });
        }

        /// <summary>
        /// 保存分类
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductCategoryInfo() {
            var req = RequestParm;
            RB_Product_Category_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Category_Extend>(req.msg.ToString());
            if ((demodel.Tier ?? 0) <= 0) {
                return ApiResult.ParamIsNull("请选择层级");
            }
            if (string.IsNullOrEmpty(demodel.Name)) {
                return ApiResult.ParamIsNull("请输入分类名称");
            }
            if (!demodel.Sort.HasValue) {
                return ApiResult.ParamIsNull("请输入排序");
            }
            if (demodel.Tier == 1)
            {
                demodel.ParentId = 0;
            }
            else if (demodel.Tier > 1) {
                if ((demodel.ParentId ?? 0) <= 0) {
                    return ApiResult.ParamIsNull("请选择上级");
                }
            }

            demodel.TenantId = Convert.ToInt32(req.uid);
            demodel.MallBaseId = req.MallBaseId;
            demodel.Icon ??= 0;
            demodel.BigIcon ??= 0;
            demodel.AdvertisingImage ??= 0;
            demodel.AdvertisingAddress ??= "";
            demodel.Enabled ??= 1;
            demodel.IsShow ??= 1;
            demodel.Status ??= 0;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            bool flag= productModule.SetProductCategoryInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }

        }

        /// <summary>
        /// 删除分类
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelProductCategoryInfo() {
            var req = RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int CategoryId = parms.GetInt("CategoryId", 0);
            if (CategoryId <= 0)
            {
                return ApiResult.ParamIsNull("请选择需删除分类id");
            }

            bool flag = productModule.DelProductCategoryInfo(CategoryId, req.TenantId, req.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 保存分类转移
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductCategoryTransfer() {
            var req = RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int CategoryId = parms.GetInt("CategoryId", 0);
            int ParentId = parms.GetInt("ParentId", 0);
            if (CategoryId <= 0) {
                return ApiResult.ParamIsNull("请选择需转移分类id");
            }
            if (ParentId < 0) {
                return ApiResult.ParamIsNull("请选择转入分类id");
            }
            if (CategoryId == ParentId) {
                return ApiResult.ParamIsNull("不能转入同一分类下");
            }

            var list = productModule.GetProductCategoryForALLChlid(CategoryId, Convert.ToInt32(req.uid), req.MallBaseId);
            if (list.Where(x => x.Id == ParentId).Any()) {
                return ApiResult.ParamIsNull("不能转入当前分类的子集分类");
            }
            var model = list.Where(x => x.Id == CategoryId).FirstOrDefault();
            if (model == null) {
                return ApiResult.ParamIsNull("分类不存在，请核实后再试");
            }
            if (model.ParentId == ParentId) {
                return ApiResult.ParamIsNull("已在当前分类下，未转移");
            }

            bool flag = productModule.SetProductCategoryTransfer(model, ParentId, list);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 保存分类排序
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductCategorySort() {
            var req = RequestParm;
            List<RB_Product_CategorySort> delist = JsonConvert.DeserializeObject<List<RB_Product_CategorySort>>(req.msg.ToString());
            if (delist == null || !delist.Any()) {
                return ApiResult.ParamIsNull();
            }

            bool flag = productModule.SetProductCategorySort(delist, req.TenantId, req.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 获取分类样式
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductCategoryStyleInfo() {
            var parms = RequestParm;

            var model = productModule.GetProductCategoryStyleInfo(parms.TenantId, parms.MallBaseId);
            return ApiResult.Success("", new
            {
                model.Id,
                model.Type,
                model.Style,
                model.CategoryNum,
                model.RowNum,
                UpdateDate = model.UpdateDate.Value.ToString("yyyy-MM-dd HH:mm:ss")
            });
        }

        /// <summary>
        /// 保存分类样式
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductCategoryStyleInfo() {
            var req = RequestParm;
            RB_Product_CategoryStyle_Extend demodel = JsonConvert.DeserializeObject<RB_Product_CategoryStyle_Extend>(req.msg.ToString());

            demodel.CategoryNum ??= 1;
            demodel.CreateDate = DateTime.Now;
            demodel.MallBaseId = req.MallBaseId;
            demodel.RowNum ??= 1;
            demodel.Style ??= 1;
            demodel.TenantId = req.TenantId;
            demodel.Type ??= 1;
            demodel.UpdateDate = DateTime.Now;
            bool flag = productModule.SetProductCategoryStyleInfo(demodel);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        #endregion

        #region 商品规格

        /// <summary>
        /// 获取商品规格分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductSpecificationPageList()
        {
            var parms = RequestParm;
            ResultPageModel pagelist = JsonConvert.DeserializeObject<ResultPageModel>(parms.msg.ToString());
            RB_Product_Specification_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Specification_Extend>(parms.msg.ToString());

            demodel.TenantId = parms.TenantId;
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProducSpecificationPageList(pagelist.pageIndex, pagelist.pageSize, out long count, demodel);
            pagelist.count = Convert.ToInt32(count);
            pagelist.pageData = list.Select(x => new
            {
                x.Id,
                x.Name,
                x.SpecList,
                x.TenantId,
                x.MallBaseId,
                UpdateDate = x.UpdateDate.HasValue ? x.UpdateDate.Value.ToString("yyyy-MM-dd") : ""
            });
            return ApiResult.Success("", pagelist);
        }

        /// <summary>
        /// 获取规格列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductSpecificationList()
        {
            var parms = RequestParm;            
            RB_Product_Specification_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Specification_Extend>(parms.msg.ToString());

            demodel.TenantId = parms.TenantId;
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProducSpecificationList(demodel);
            return ApiResult.Success("", list.Select(x => new
            {
                x.Id,
                x.Name,
                x.SpecList,
                x.TenantId,
                x.MallBaseId,
                UpdateDate = x.UpdateDate.HasValue ? x.UpdateDate.Value.ToString("yyyy-MM-dd") : ""
            }));
        }

        /// <summary>
        /// 保存修改规格
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductSpecificationInfo() {
            var req = RequestParm;
            RB_Product_Specification_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Specification_Extend>(req.msg.ToString());
            if (string.IsNullOrEmpty(demodel.Name)) {
                return ApiResult.ParamIsNull("请输入规格名");
            }
            if (demodel.SpecList == null || !demodel.SpecList.Any()) {
                return ApiResult.ParamIsNull("请输入规格值");
            }
            demodel.Content = JsonConvert.SerializeObject(demodel.SpecList);
            demodel.TenantId = req.TenantId;
            demodel.MallBaseId = req.MallBaseId;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            demodel.Status = 0;
            bool flag = productModule.SetProductSpecificationInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 删除商品规格
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelProductSpecificationInfo() {
            var req = RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int SpecificationId = parms.GetInt("SpecificationId", 0);
            if (SpecificationId <= 0) {
                return ApiResult.ParamIsNull();
            }

            bool flag = productModule.DelProductSpecificationInfo(SpecificationId, req.TenantId, req.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }
        }

        #endregion

        #region 快速购买
        /// <summary>
        /// 获取商品快速购买分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductQuickBuyPageList()
        {
            var parms = RequestParm;
            ResultPageModel pagelist = JsonConvert.DeserializeObject<ResultPageModel>(parms.msg.ToString());
            RB_Product_QuickBuy_Extend demodel = JsonConvert.DeserializeObject<RB_Product_QuickBuy_Extend>(parms.msg.ToString());

            demodel.TenantId = parms.TenantId;
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProductQuickBuyPageList(pagelist.pageIndex, pagelist.pageSize, out long count, demodel);
            pagelist.count = Convert.ToInt32(count);
            pagelist.pageData = list.Select(x => new
            {
                x.Id,
                x.CategoryId,
                x.CategoryName,
                x.Icon,
                x.IconPath,
                x.Sort,
                x.TenantId,
                x.MallBaseId,
                CreateDate = x.CreateDate.HasValue ? x.CreateDate.Value.ToString("yyyy-MM-dd HH:mm:ss") : ""
            });
            return ApiResult.Success("", pagelist);
        }

        /// <summary>
        /// 获取快速购买列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductQuickBuyList()
        {
            var parms = RequestParm;
            RB_Product_QuickBuy_Extend demodel = JsonConvert.DeserializeObject<RB_Product_QuickBuy_Extend>(parms.msg.ToString());

            demodel.TenantId = parms.TenantId;
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProductQuickBuyList(demodel);
            return ApiResult.Success("", list.Select(x => new
            {
                x.Id,
                x.CategoryId,
                x.CategoryName,
                x.Icon,
                x.IconPath,
                x.Sort,
                x.TenantId,
                x.MallBaseId
            }));
        }

        /// <summary>
        /// 保存修改快速购买
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductQuickBuyInfo()
        {
            var req = RequestParm;
            RB_Product_QuickBuy_Extend demodel = JsonConvert.DeserializeObject<RB_Product_QuickBuy_Extend>(req.msg.ToString());
            if ((demodel.CategoryId??0)<=0)
            {
                return ApiResult.ParamIsNull("请选择分类");
            }
            if (!demodel.Sort.HasValue)
            {
                return ApiResult.ParamIsNull("请输入排序");
            }
            if (productModule.ValidateQuickBuyCategory(demodel.Id, demodel.CategoryId, req.TenantId, req.MallBaseId)) {
                return ApiResult.Failed("该分类已存在，无法再次添加");
            }

            demodel.TenantId = req.TenantId;
            demodel.MallBaseId = req.MallBaseId;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            demodel.Status = 0;
            bool flag = productModule.SetProductQuickBuyInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 删除商品快速购买
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelProductQuickBuyInfo()
        {
            var req = RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int QuickBuyId = parms.GetInt("QuickBuyId", 0);
            if (QuickBuyId <= 0)
            {
                return ApiResult.ParamIsNull();
            }

            bool flag = productModule.DelProductQuickBuyInfo(QuickBuyId, req.TenantId, req.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }


        #endregion

        #region 推荐设置

        /// <summary>
        /// 获取推荐设置详情
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductRecommendSetInfo() {
            var req = RequestParm;
            var model = productModule.GetProductRecommendSetInfo(req.TenantId, req.MallBaseId);
            return ApiResult.Success("", new
            {
                model.Id,
                model.DetailRecommendStatus,
                model.DetailRecommendNum,
                model.OrderCompleteStatus,
                model.IsOrderRecommend,
                model.OrderProductIds,
                model.OrderProductList,
                model.CommentRecommentStatus,
                model.IsCommentRecommend,
                model.CommentProductIds,
                model.CommentProductList
            });
        }

        /// <summary>
        /// 保存推荐设置
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductRecommendSetInfo() {
            var req = RequestParm;
            RB_Product_Recommend_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Recommend_Extend>(req.msg.ToString());

            if (demodel.IsOrderRecommend == 1) {
                if (demodel.OrderProductList == null || !demodel.OrderProductList.Any()) {
                    return ApiResult.ParamIsNull("请选择自定义推荐商品");
                }
                demodel.OrderProductIds = string.Join(",", demodel.OrderProductList.Select(x => x.Id).Distinct());
            }
            if (demodel.IsCommentRecommend == 1) {
                if (demodel.CommentProductList == null || !demodel.CommentProductList.Any()) {
                    return ApiResult.ParamIsNull("请选择自定义推荐商品");
                }
                demodel.CommentProductIds = string.Join(",", demodel.CommentProductList.Select(x => x.Id).Distinct());
            }

            demodel.TenantId = req.TenantId;
            demodel.MallBaseId = req.MallBaseId;
            demodel.OrderProductIds ??= "";
            demodel.CommentProductIds ??= "";
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            bool flag = productModule.SetProductRecommendSetInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed();
            }
        }

        #endregion

        #region 商品服务

        /// <summary>
        /// 获取商品服务分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductServicePageList()
        {
            var parms = RequestParm;
            ResultPageModel pagelist = JsonConvert.DeserializeObject<ResultPageModel>(parms.msg.ToString());
            RB_Product_Service_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Service_Extend>(parms.msg.ToString());

            demodel.TenantId = parms.TenantId;
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProductServicePageList(pagelist.pageIndex, pagelist.pageSize, out long count, demodel);
            pagelist.count = Convert.ToInt32(count);
            pagelist.pageData = list.Select(x => new
            {
                x.Id,
                x.Name,
                x.Sort,
                x.Remark,
                x.IsDefault,
                x.TenantId,
                x.MallBaseId,
                UpdateDate = x.UpdateDate.HasValue ? x.UpdateDate.Value.ToString("yyyy-MM-dd") : ""
            });
            return ApiResult.Success("", pagelist);
        }

        /// <summary>
        /// 获取服务列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetProductServiceList()
        {
            var parms = RequestParm;
            RB_Product_Service_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Service_Extend>(parms.msg.ToString());

            demodel.TenantId = parms.TenantId;
            demodel.MallBaseId = parms.MallBaseId;
            var list = productModule.GetProductServiceList(demodel);
            return ApiResult.Success("", list.Select(x => new
            {
                x.Id,
                x.Name,
                x.Sort,
                x.Remark,
                x.IsDefault
            }));
        }

        /// <summary>
        /// 保存修改服务
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetProductServiceInfo()
        {
            var req = RequestParm;
            RB_Product_Service_Extend demodel = JsonConvert.DeserializeObject<RB_Product_Service_Extend>(req.msg.ToString());
            if (string.IsNullOrEmpty(demodel.Name))
            {
                return ApiResult.ParamIsNull("请输入服务名称");
            }
            if (!demodel.Sort.HasValue)
            {
                return ApiResult.ParamIsNull("请输入排序");
            }
            
            demodel.TenantId = req.TenantId;
            demodel.MallBaseId = req.MallBaseId;
            demodel.IsDefault ??= 1;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateDate = DateTime.Now;
            demodel.Status = 0;
            bool flag = productModule.SetProductServiceInfo(demodel);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        /// <summary>
        /// 删除商品服务
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelProductServiceInfo()
        {
            var req = RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int ServiceId = parms.GetInt("ServiceId", 0);
            if (ServiceId <= 0)
            {
                return ApiResult.ParamIsNull();
            }

            bool flag = productModule.DelProductServiceInfo(ServiceId, req.TenantId, req.MallBaseId);
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        #endregion

        #region 商品列表



        #endregion
    }
}