﻿using System;
using System.Collections.Generic;
using System.Linq;
using Mall.CacheManager.User;
using Mall.Common.API;
using Mall.Common.Plugin;
using Mall.Model.Extend.TradePavilion;
using Mall.Module.TradePavilion;
using Mall.WebApi.Filter;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Cors;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Mall.WebApi.Controllers.TradePavilion
{
    [Route("api/[controller]/[action]")]
    [ApiExceptionFilter]
    [ApiController]
    [EnableCors("AllowCors")]
    public class TradeController : BaseController
    {
        /// <summary>
        /// 联系人处理类对象
        /// </summary>
        private readonly ContractModule module = new ContractModule();

        /// <summary>
        /// 投资采购处理类对象
        /// </summary>
        private readonly InvestmentProcurementModule procurementModule = new InvestmentProcurementModule();

        /// <summary>
        /// 公司信息处理类对象
        /// </summary>
        private readonly CompanyModule companyModule = new CompanyModule();


        /// <summary>
        /// 商会活动处理类对象
        /// </summary>
        private readonly CommerceActivityModule activityModule = new CommerceActivityModule();

        /// <summary>
        /// 商会活动报名处理类对象
        /// </summary>
        private readonly CommerceConsultModule consultModule = new CommerceConsultModule();

        /// <summary>
        /// 韩国馆新闻处理类
        /// </summary>
        private readonly CommerceNewsModule commerceNewsModule = new CommerceNewsModule();
        /// <summary>
        /// 韩国馆新闻处理类
        /// </summary>
        private readonly CarrierModule carrierModule = new CarrierModule();
        /// <summary>
        /// 认证对象
        /// </summary>
        private readonly AuthenticationModule authenticationModule = new AuthenticationModule();

        /// <summary>
        /// Fat资料处理类对象
        /// </summary>
        private readonly CommerceAskModule askModule = new CommerceAskModule();

        /// <summary>
        /// 会员信息
        /// </summary>
        private readonly Module.User.UserModule userModule = new Module.User.UserModule();
        /// <summary>
        /// 榜单
        /// </summary>
        private readonly FirstShopListModule firstShopListModule = new FirstShopListModule();
        /// <summary>
        /// 心愿单
        /// </summary>
        private readonly FirstShopWishModule firstShopWishModule = new FirstShopWishModule();



        #region 联系人管理

        /// <summary>
        /// 获取联系人分页列表
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        [AllowAnonymous]
        public ApiResult GetContractPageList()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Contract_Extend()
            {
                Name = parms.GetStringValue("Name")
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = module.GetContractPageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            if (list != null && list.Count > 0)
            {
                foreach (var item in list)
                {
                    if (item.CreateBy > 0)
                    {
                        item.CreateByName = UserReidsCache.GetUserLoginInfo(item.CreateBy)?.Account ?? "";
                    }
                }
            }
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }

        /// <summary>
        /// 获取联系人列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetContractList()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Contract_Extend()
            {
                Name = parms.GetStringValue("Name")
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = module.GetContractListModule(query);
            if (list != null && list.Count > 0)
            {
                foreach (var item in list)
                {
                    if (item.CreateBy > 0)
                    {
                        item.CreateByName = UserReidsCache.GetUserLoginInfo(item.CreateBy)?.Account ?? "";
                    }
                }
            }
            return ApiResult.Success(data: list);
        }

        /// <summary>
        /// 新增修改联系人
        /// </summary>
        /// <returns></returns>
        public ApiResult SetContract()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var extModel = new RB_Contract_Extend()
            {
                Id = parms.GetInt("Id"),
                Name = parms.GetStringValue("Name"),
                Tel = parms.GetStringValue("Tel"),
                HeadIcon = parms.GetStringValue("HeadIcon"),
                BgImg = parms.GetStringValue("BgImg"),
                Position = parms.GetStringValue("Position"),
                Intro = parms.GetStringValue("Intro")
            };
            extModel.Status = Common.Enum.DateStateEnum.Normal;
            extModel.MallBaseId = RequestParm.MallBaseId;
            extModel.TenantId = RequestParm.TenantId;
            Int32.TryParse(RequestParm.uid, out int Uid);
            extModel.CreateBy = Uid;
            extModel.CreateDate = DateTime.Now;
            extModel.UpdateBy = Uid;
            extModel.UpdateDate = DateTime.Now;
            var flag = module.SetContractModule(extModel);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 获取角色实体
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        [AllowAnonymous]
        public ApiResult GetContract()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = module.GetContractModule(Id);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 删除联系人
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveContract()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = module.RemoveContractModule(Id, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        #endregion

        #region 投资采购管理

        /// <summary>
        /// 获取投资采购分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetInvestmentProcurementPage()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_InvestmentProcurement_Extend()
            {
                CompanyName = parms.GetStringValue("CompanyName"),
                Type = parms.GetInt("Type"),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = procurementModule.GetInvestmentProcurementPageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }



        /// <summary>
        /// 获取投资采购详情
        /// </summary>
        /// <returns></returns>
        public ApiResult GetInvestmentProcurement()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = procurementModule.GetInvestmentProcurementModule(Id);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 删除投资采购信息
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveInvestmentProcurement()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = procurementModule.RemoveInvestmentProcurementModule(Id, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 投资采购审核
        /// </summary>
        /// <returns></returns>
        public ApiResult AutitInvestmentProcurement()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            Int32.TryParse(RequestParm.uid, out int Uid);
            var extModel = new RB_InvestmentProcurement_Extend()
            {
                Id = parms.GetInt("Id"),
                AuditStatus = parms.GetInt("AuditStatus"),
                AuditTime = DateTime.Now,
                AuditBy = Uid,
                AuditContent = parms.GetStringValue("AuditContent")
            };
            var flag = procurementModule.AuditInvestmentProcurementModule(extModel);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }
        #endregion

        #region 公司信息管理

        /// <summary>
        /// 获取公司信息分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCompanyPageList()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Company_Extend()
            {
                CompanyName = parms.GetStringValue("CompanyName"),
                CompanyStatus = parms.GetInt("CompanyStatus", -1),
                CreateBy = parms.GetInt("CreateBy", 0),
                IsComplete = 1//需是 已完善了资料的
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = companyModule.GetCompanyPageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }

        /// <summary>
        /// 获取公司信息详情
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCompany()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var CompanyId = parms.GetInt("CompanyId", 0);
            var extModel = companyModule.GetCompanyModule(CompanyId);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 删除公司信息
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveCompany()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var CompanyId = parms.GetInt("CompanyId", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = companyModule.RemoveCompanyModule(CompanyId, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 审核公司信息
        /// </summary>
        /// <returns></returns>
        public ApiResult UpdateCompanyStatus()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var CompanyId = parms.GetInt("CompanyId", 0);
            var Status = parms.GetInt("CompanyStatus", 0);
            var Remark = parms.GetStringValue("Remark");//不通过备注
            var model = companyModule.GetCompanyEntity(CompanyId);
            if (model == null || model.CompanyId <= 0)
            {
                return ApiResult.Failed("公司信息不存在");
            }
            if (Status != 1 && Status != 2)
            {
                return ApiResult.Failed("状态有误");
            }
            if (Status == 2 && string.IsNullOrEmpty(Remark))
            {
                return ApiResult.ParamIsNull("请输入不通过的原因");
            }
            if (model.CompanyStatus == 1)
            {
                return ApiResult.Failed("公司信息已审核通过，请勿重复审核");
            }
            var msg = companyModule.UpdateCompanyStatus(CompanyId, Status, Remark);
            if (msg == "" && Status > 0)//推送审核结果
            {
                //查询用户的信息
                var umodel = userModule.GetMemberUserInfo(model.CreateBy);
                if (umodel != null && !string.IsNullOrWhiteSpace(umodel.OpenId))
                {
                    DateTime dt1 = DateTime.Now;
                    List<string> openidList = new List<string>
                    {
                        umodel.OpenId
                    };
                    new Module.User.MiniProgramMsgModule().SendHatchExamineMsg(model.TenantId, model.MallBaseId, openidList, Status == 1 ? "认证通过" : "认证失败", "企业认证结果通知", System.DateTime.Now.ToString("yyyy年MM月dd日 HH:mm"));
                    DateTime dt2 = DateTime.Now;
                    LogHelper.Write("认证通知耗时：" + (dt2 - dt1).TotalSeconds);
                }
            }
            return msg == "" ? ApiResult.Success() : ApiResult.Failed(msg);
        }

        /// <summary>
        /// 设置用户为管理员
        /// </summary>
        /// <returns></returns>
        public ApiResult SetMemberUserIsAdmin()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var UserId = parms.GetInt("UserId", 0);
            var IsAdmin = parms.GetInt("IsAdmin", 0);
            if (UserId <= 0)
            {
                return ApiResult.Failed("请传递用户id");
            }

            var msg = companyModule.SetMemberUserIsAdmin(UserId, IsAdmin);
            if (msg == "")
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed(msg);
            }
        }
        #endregion

        #region 商会活动管理

        /// <summary>
        /// 获取商会活动分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceActivityPage()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Commerce_Activity_Extend()
            {
                ActivityName = jObj.GetStringValue("ActivityName")
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = activityModule.GetCommerceActivityPageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query, isQuerySignup: true);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }

        /// <summary>
        /// 新增修改商会活动
        /// </summary>
        /// <returns></returns>
        public ApiResult SetCommerceActivity()
        {
            JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            var extModel = new RB_Commerce_Activity_Extend()
            {
                Id = jObj.GetInt("Id"),
                ActivityName = jObj.GetStringValue("ActivityName"),
                StartTime = jObj.GetDateTime("StartTime"),
                EndTime = jObj.GetDateTime("EndTime"),
                CoverImg = jObj.GetStringValue("CoverImg"),
                ActivityType = jObj.GetInt("ActivityType"),
                ActivityInfo = jObj.GetStringValue("ActivityInfo"),
                SignStartTime = jObj.GetDateTime("SignStartTime"),
                SignEndTime = jObj.GetDateTime("SignEndTime"),
                IsFree = jObj.GetInt("IsFree", 1),
                Price = jObj.GetInt("Price", 0),
                IsCancelConfirm = jObj.GetInt("IsCancelConfirm", 2)
            };
            try
            {
                extModel.CommerceDetailsList = Common.Plugin.JsonHelper.DeserializeObject<List<RB_Commerce_Details_Extend>>(jObj.GetStringValue("CommerceDetailsList"));
            }
            catch (Exception ex)
            {
                Common.Plugin.LogHelper.Write(ex, "SetCommerceActivity_CommerceDetailsList");
            }
            extModel.Status = Common.Enum.DateStateEnum.Normal;
            extModel.MallBaseId = RequestParm.MallBaseId;
            extModel.TenantId = RequestParm.TenantId;
            Int32.TryParse(RequestParm.uid, out int Uid);
            extModel.CreateBy = Uid;
            extModel.CreateDate = DateTime.Now;
            extModel.UpdateBy = Uid;
            extModel.UpdateDate = DateTime.Now;
            if (extModel.IsFree == 1)
            {
                extModel.Price = 0;
            }
            var flag = activityModule.SetCommerceActivityModule(extModel);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 获取商会活动详情
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceActivity()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = activityModule.GetCommerceActivityModule(Id);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 更新商户活动状态
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveCommerceActivity()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = activityModule.RemoveCommerceActivityModule(Id, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 获取商会活动类型分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceActivityTypePage()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Commerce_ActivityType_Extend()
            {
                TypeName = jObj.GetStringValue("TypeName")
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = activityModule.GetCommerceActivityTypePageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            list.ForEach(x => x.LableNameList = string.IsNullOrWhiteSpace(x.LableName) ? new List<string>() : JsonConvert.DeserializeObject<List<string>>(x.LableName));
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }

        /// <summary>
        /// 获取商会活动类型列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceActivityTypeList()
        {
            JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Commerce_ActivityType_Extend()
            {
                TypeName = jObj.GetStringValue("TypeName")
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = activityModule.GetCommerceActivityTypeListModule(query);
            list.ForEach(x => x.LableNameList = string.IsNullOrWhiteSpace(x.LableName) ? new List<string>() : JsonConvert.DeserializeObject<List<string>>(x.LableName));
            return ApiResult.Success(data: list);
        }

        /// <summary>
        /// 新增修改商会活动类型
        /// </summary>
        /// <returns></returns>
        public ApiResult SetCommerceActivityType()
        {
            RB_Commerce_ActivityType_Extend extModel = JsonConvert.DeserializeObject<RB_Commerce_ActivityType_Extend>(RequestParm.msg.ToString());
            //JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            //var extModel = new RB_Commerce_ActivityType_Extend()
            //{
            //    Id = jObj.GetInt("Id"),
            //    TypeName = jObj.GetStringValue("TypeName"),
            //    CoverImage=jObj.GetStringValue("CoverImage")
            //};
            extModel.Status = Common.Enum.DateStateEnum.Normal;
            extModel.MallBaseId = RequestParm.MallBaseId;
            extModel.TenantId = RequestParm.TenantId;
            Int32.TryParse(RequestParm.uid, out int Uid);
            extModel.CreateBy = Uid;
            extModel.CreateDate = DateTime.Now;
            extModel.UpdateBy = Uid;
            extModel.UpdateDate = DateTime.Now;
            if (extModel.LableNameList != null && extModel.LableNameList.Any())
            {
                extModel.LableName = JsonConvert.SerializeObject(extModel.LableNameList);
            }
            else
            {
                extModel.LableName = "";
            }
            var flag = activityModule.SetCommerceActivityTypeModule(extModel);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 获取商会活动类型详情
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceActivityType()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = activityModule.GetCommerceActivityTypeModule(Id);
            extModel.LableNameList = string.IsNullOrWhiteSpace(extModel.LableName) ? new List<string>() : JsonConvert.DeserializeObject<List<string>>(extModel.LableName);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 更新商户活动类型状态
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveCommerceActivityType()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = activityModule.RemoveCommerceActivityTypeModule(Id, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }
        #endregion

        #region 商会活动报名管理

        /// <summary>
        /// 获取商会活动报名分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceConsultPage()
        {
            JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Commerce_Consult_Extend()
            {
                ActivityId = jObj.GetInt("ActivityId", 0),
                LinkMan = jObj.GetStringValue("LinkMan"),
                EnrollState = jObj.GetInt("EnrollState", 0)
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = consultModule.GetCommerceConsultListModule(query);
            return ApiResult.Success(data: list);
        }



        /// <summary>
        /// 根据编号获取商会活动报名实体
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceConsult()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = consultModule.GetCommerceConsultModule(Id);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 更新商会活动报名状态
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveCommerceConsult()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = consultModule.RemoveCommerceConsultModule(Id, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 操作商会活动报名申请取消
        /// </summary>
        /// <returns></returns>
        public ApiResult SetCommerceConsultApplyCancel()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var ApplyForCancelStatus = parms.GetInt("ApplyForCancelStatus", 0);//申请取消审核状态  1同意  2拒绝
            var RejectRemark = parms.GetStringValue("RejectRemark");//拒绝备注
            if (Id <= 0)
            {
                return ApiResult.ParamIsNull("请传递");
            }
            if (ApplyForCancelStatus == 2)
            {
                if (string.IsNullOrEmpty(RejectRemark))
                {
                    return ApiResult.ParamIsNull("请输入拒绝的原因");
                }
            }
            var flag = consultModule.SetCommerceConsultApplyCancel(Id, ApplyForCancelStatus, RejectRemark);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }
        #endregion

        #region 新闻信息管理

        /// <summary>
        /// 获取新闻信息分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceNewsPage()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Commerce_News_Extend()
            {
                Title = parms.GetStringValue("Title"),
                Type = parms.GetInt("Type"),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = commerceNewsModule.GetCommerceNewsPageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }


        /// <summary>
        /// 获取新闻信息详情
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceNews()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = commerceNewsModule.GetCommerceNewsModule(Id);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 删除新闻信息
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveCommerceNews()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = commerceNewsModule.RemoveCommerceNewsModule(Id, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 新增修改新闻
        /// </summary>
        /// <returns></returns>
        public ApiResult SetCommerceNews()
        {
            JObject jObj = JObject.Parse(RequestParm.msg.ToString());
            var extModel = new RB_Commerce_News_Extend()
            {
                Id = jObj.GetInt("Id"),
                CoverImg = jObj.GetStringValue("CoverImg"),
                Title = jObj.GetStringValue("Title"),
                Industry = jObj.GetStringValue("Industry"),
                PublishTime = jObj.GetDateTime("PublishTime"),
                NewsInfo = jObj.GetStringValue("NewsInfo"),
                Type = jObj.GetInt("Type"),
                Subtitle = jObj.GetStringValue("Subtitle"),
            };
            extModel.Status = Common.Enum.DateStateEnum.Normal;
            extModel.MallBaseId = RequestParm.MallBaseId;
            extModel.TenantId = RequestParm.TenantId;
            Int32.TryParse(RequestParm.uid, out int Uid);
            extModel.CreateBy = Uid;
            extModel.CreateDate = DateTime.Now;
            extModel.UpdateBy = Uid;
            extModel.UpdateDate = DateTime.Now;
            var flag = commerceNewsModule.SetCommerceNewsModule(extModel);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }
        #endregion

        #region 咨询管理
        /// <summary>
        /// 获取投资采购分页列表
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceAskPage()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Commerce_Ask_Extend()
            {
                CompanyName = parms.GetStringValue("CompanyName"),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = askModule.GetCommerceAskPageListModule(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }

        /// <summary>
        /// 获取新闻信息详情
        /// </summary>
        /// <returns></returns>
        public ApiResult GetCommerceAskInfo()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var extModel = askModule.GetCommerceAskModule(Id);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 获取新闻信息详情
        /// </summary>
        /// <returns></returns>
        public ApiResult ReplyCommerceAsk()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            Int32.TryParse(RequestParm.uid, out int Uid);
            var model = new RB_Commerce_Ask_Extend()
            {
                Id = parms.GetInt("Id", 0),
                AskStatus = parms.GetInt("AskStatus"),
                ReplyTime = DateTime.Now,
                FatContent = parms.GetStringValue("FatContent"),
                ReplyId = Uid
            };
            var extModel = askModule.ReplyCommerceAskModule(model);
            return ApiResult.Success(data: extModel);
        }
        #endregion

        #region 认证管理

        /// <summary>
        /// 获取认证分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetAuthenticationPageList()
        {
            var parms = RequestParm;
            var userInfo = this.UserInfo;
            ResultPageModel pagelist = JsonConvert.DeserializeObject<ResultPageModel>(parms.msg.ToString());
            RB_Authentication_Extend demodel = JsonConvert.DeserializeObject<RB_Authentication_Extend>(parms.msg.ToString());
            demodel.TenantId = userInfo.TenantId;
            demodel.MallBaseId = userInfo.MallBaseId;
            var list = authenticationModule.GetAuthenticationPageList_V2(pagelist.pageIndex, pagelist.pageSize, out long count, demodel);
            pagelist.count = Convert.ToInt32(count);
            pagelist.pageData = list.Select(x => new
            {
                x.Id,
                x.Name,
                x.CoverImage,
                x.ImageList,
                x.VideoUrl,
                x.Type,
                TypeName = x.Type.GetEnumName(),
                x.BrandType,
                BrandTypeName = x.BrandType.GetEnumName(),
                OpenTime = x.OpenTime.ToString("yyyy-MM-dd HH:mm"),
                x.Address,
                x.TradingArea,
                x.BusinessType,
                BusinessTypeName = x.BusinessType.GetEnumName(),
                x.BusinessDetail,
                x.Description,
                x.County,
                x.Category,
                CategoryName = x.Category.GetEnumName(),
                x.Contact,
                x.ContactMobile,
                x.AuditState,
                x.AuditRemark,
                AuditDate = x.AuditDate.HasValue ? x.AuditDate.Value.ToString("yyyy-MM-dd HH:mm") : "",
                CreateDate = x.CreateDate.ToString("yyyy-MM-dd HH:mm"),
                x.CreateName,
                x.CreatePhoto
            });
            return ApiResult.Success("", pagelist);
        }

        /// <summary>
        /// 认证Excel下载
        /// </summary>
        [HttpPost]
        public FileContentResult GetAuthenticationListToExcel()
        {
            var userInfo = this.UserInfo;
            RB_Authentication_Extend demodel = JsonConvert.DeserializeObject<RB_Authentication_Extend>(RequestParm.msg.ToString());
            string ExcelName = "首店申请" + DateTime.Now.ToString("yyyyMMddHHmmss") + ".xls";
            List<ExcelDataSource> slist = new List<ExcelDataSource>();
            ExcelDataSource header = new ExcelDataSource()
            {
                ExcelRows = new List<ExcelColumn>(30) {
                    new ExcelColumn(value: "首店名称") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "封面图") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "图片") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "视频") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "类型") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "品牌归属地") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "开业时间") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "详细地址") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "商圈") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "行业") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "首店业种类型") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "简介") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "所在区县") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "类别") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "联系人") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "联系电话") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "审核状态") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                    new ExcelColumn(value: "申请时间") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER },
                }
            };

            slist.Add(header);

            try
            {
                demodel.TenantId = userInfo.TenantId;
                demodel.MallBaseId = userInfo.MallBaseId;
                var list = authenticationModule.GetAuthenticationPageList_V2(1, 100000, out long count, demodel);
                #region 组装数据
                foreach (var item in list)
                {
                    string AuditState = "审核中";
                    if (item.AuditState == 2)
                    {
                        AuditState = "通过";
                    }
                    else if (item.AuditState == 3)
                    {
                        AuditState = "不通过";
                    }
                    ExcelDataSource datarow = new ExcelDataSource()
                    {
                        ExcelRows = new List<ExcelColumn>(30) {
                            new ExcelColumn(value:item.Name){ },
                            new ExcelColumn(value:item.CoverImage){ },
                            new ExcelColumn(value:item.Images){ },
                            new ExcelColumn(value:item.VideoUrl){ },
                            new ExcelColumn(value:item.Type.GetEnumName()){ },
                            new ExcelColumn(value:item.BrandType.GetEnumName()){ },
                            new ExcelColumn(value:item.OpenTime.ToString("yyyy-MM-dd HH:mm")){ },
                            new ExcelColumn(value:item.Address){ },
                            new ExcelColumn(value:item.TradingArea){ },
                            new ExcelColumn(value:item.BusinessType.GetEnumName()){ },
                            new ExcelColumn(value:item.BusinessDetail){ },
                            new ExcelColumn(value:item.Description){ },
                            new ExcelColumn(value:item.County){ },
                            new ExcelColumn(value:item.Category.GetEnumName()){ },
                            new ExcelColumn(value:item.Contact){ },
                            new ExcelColumn(value:item.ContactMobile){ },
                            new ExcelColumn(value:AuditState){ },
                            new ExcelColumn(value:item.CreateDate.ToString("yyyy-MM-dd HH:mm")){ }
                        }
                    };
                    slist.Add(datarow);
                }
                #endregion

                var byteData = ExcelTempLateHelper.ToExcelExtend(slist);

                return File(byteData, "application/octet-stream", ExcelName);
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, string.Format("GetFileFromWebApi_requestData: {0}", JsonHelper.Serialize(RequestParm)));
                var byteData1 = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData1, "application/octet-stream", ExcelName);
            }

        }

        /// <summary>
        /// 获取认证详情
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetAuthenticationInfo()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            int AuthenticationId = parms.GetInt("AuthenticationId", 0);
            if (AuthenticationId <= 0)
            {
                return ApiResult.ParamIsNull("请传递认证id");
            }
            var model = authenticationModule.GetAuthenticationInfo(AuthenticationId);
            if (model == null) { return ApiResult.Failed("认证不存在"); }
            return ApiResult.Success("", new
            {
                model.Id,
                model.Name,
                model.CoverImage,
                model.ImageList,
                model.VideoUrl,
                model.Type,
                TypeName = model.Type.GetEnumName(),
                model.BrandType,
                BrandTypeName = model.BrandType.GetEnumName(),
                OpenTime = model.OpenTime.ToString("yyyy-MM-dd HH:mm"),
                model.Address,
                model.TradingArea,
                model.BusinessType,
                BusinessTypeName = model.BusinessType.GetEnumName(),
                model.BusinessDetail,
                model.Description,
                model.County,
                model.Category,
                CategoryName = model.Category.GetEnumName(),
                model.Contact,
                model.ContactMobile,
                model.AuditState,
                model.AuditRemark,
                AuditDate = model.AuditDate.HasValue ? model.AuditDate.Value.ToString("yyyy-MM-dd HH:mm") : "",
                CreateDate = model.CreateDate.ToString("yyyy-MM-dd HH:mm")
            });
        }

        /// <summary>
        /// 认证审核
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetAuthenticationAudit()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            int AuthenticationId = parms.GetInt("AuthenticationId", 0);
            int AuditState = parms.GetInt("AuditState", 0);//审核状态  2审核通过   3审核不通过
            string AuditRemark = parms.GetStringValue("AuditRemark");

            if (AuthenticationId <= 0)
            {
                return ApiResult.ParamIsNull("请传递认证id");
            }
            if (AuditState != 2 && AuditState != 3)
            {
                return ApiResult.Failed("状态有误");
            }
            bool flag = authenticationModule.SetAuthenticationAudit(AuthenticationId, AuditState, AuditRemark);
            if (flag)
            {
                //查询用户的信息
                var model = authenticationModule.GetAuthenticationInfo(AuthenticationId);
                var umodel = userModule.GetMemberUserInfo(model.CreateBy);
                if (umodel != null && !string.IsNullOrWhiteSpace(umodel.OpenId))
                {
                    List<string> openidList = new List<string>();
                    openidList.Add(umodel.OpenId);
                    new Module.User.MiniProgramMsgModule().SendHatchExamineMsg(model.TenantId, model.MallBaseId, openidList, AuditState == 2 ? "认证通过" : "认证失败", "首店认证结果通知", System.DateTime.Now.ToString("yyyy年MM月dd日 HH:mm"));
                }
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }



        /// <summary>
        /// 批量认证审核
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult BatchAuthenticationAudit()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var userInfo = this.UserInfo;
            string AuthenticationIds = parms.GetStringValue("AuthenticationIds");
            int AuditState = parms.GetInt("AuditState", 0);//审核状态  2审核通过   3审核不通过
            string AuditRemark = parms.GetStringValue("AuditRemark");

            if (string.IsNullOrWhiteSpace(AuthenticationIds))
            {
                return ApiResult.ParamIsNull("请传递认证id");
            }
            if (AuditState != 2 && AuditState != 3)
            {
                return ApiResult.Failed("状态有误");
            }

            var list = authenticationModule.GetAuthenticationList(new RB_Authentication_Extend { TenantId = userInfo.TenantId, MallBaseId = userInfo.MallBaseId, Ids = AuthenticationIds });
            if (list == null || !list.Any())
            {
                return ApiResult.Failed("请传入正确的id");
            }
            if (list.Any(x => x.AuditState > 1))
            {
                return ApiResult.Failed("已审核的不能重复审核");
            }
            bool flag = authenticationModule.BatchAuthenticationAudit(AuthenticationIds, AuditState, AuditRemark);
            if (flag)
            {
                //查询用户的信息
                string uids = string.Join(",", list.Select(x => x.CreateBy));
                var memberList = userModule.GetMemberUserList(new Model.Extend.User.RB_Member_User_Extend { TenantId = userInfo.TenantId, MallBaseId = userInfo.MallBaseId, UserIds = uids });
                if (memberList != null && memberList.Any())
                {
                    List<string> openidList = new List<string>();
                    foreach (var item in memberList)
                    {
                        if (!string.IsNullOrWhiteSpace(item.OpenId))
                        {
                            openidList.Add(item.OpenId);
                        }
                    }
                    if (openidList != null && openidList.Any())
                    {
                        new Module.User.MiniProgramMsgModule().SendHatchExamineMsg(userInfo.TenantId, userInfo.MallBaseId, openidList, AuditState == 2 ? "认证通过" : "认证失败", "首店认证结果通知", System.DateTime.Now.ToString("yyyy年MM月dd日 HH:mm"));
                    }
                }
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }



        /// <summary>
        /// 批量设置奖项
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetAuthenticationPrize()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var userInfo = this.UserInfo;
            string AuthenticationIds = parms.GetStringValue("AuthenticationIds");
            string SourceIds = parms.GetStringValue("SourceIds");
            if (string.IsNullOrWhiteSpace(AuthenticationIds))
            {
                return ApiResult.ParamIsNull("请传递认证id");
            }
            var list = authenticationModule.GetAuthenticationList(new RB_Authentication_Extend { TenantId = userInfo.TenantId, MallBaseId = userInfo.MallBaseId, Ids = AuthenticationIds });
            if (list == null || !list.Any())
            {
                return ApiResult.Failed("请传入正确的id");
            }
            if (list.Any(x => x.AuditState != 2))
            {
                return ApiResult.Failed("未审核/审核不通过的不能授予奖章");
            }
            var brandIds = string.Join(",", list.Where(x => x.ExistBrandId > 0).Select(x => x.ExistBrandId));
            bool flag = false;
            if (!string.IsNullOrWhiteSpace(brandIds))
            {
                flag = authenticationModule.SetAuthenticationPrize(brandIds, SourceIds);
            }
            
            if (flag)
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }
        #endregion

        #region 载体信息
        /// <summary>
        /// 获取载体分页列表
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        public ApiResult GetCarrierPageList()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Carrier_Extend()
            {
                MetroName = parms.GetStringValue("MetroName"),
                CarrierName = parms.GetStringValue("CarrierName"),
                MetroNum = parms.GetInt("MetroNum", 0),
                Developers = parms.GetStringValue("Developers"),
                StartOpenTime = parms.GetStringValue("StartOpenTime"),
                EndOpenTime = parms.GetStringValue("EndOpenTime"),
                ProjectType = (Common.Enum.TradePavilion.ProjectTypeEnum)parms.GetInt("ProjectType", 0),
                UserId = parms.GetInt("UserId", 0),
                OpeningStatus = parms.GetInt("OpeningStatus", -1),
                FirstStoreTest = parms.GetInt("FirstStoreTest", -1),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = carrierModule.GetCarrierPageList(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query, true);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list.Select(x => new
            {
                x.ID,
                x.CarrierName,
                x.Logo,
                CreateDate = x.CreateDate.ToString("yyyy-MM-dd HH:mm"),
                x.Address,
                x.LatAndLon,
                x.Developers,
                OpenTime = x.OpenTime.HasValue ? x.OpenTime.Value.ToString("yyyy-MM-dd HH:mm") : "",
                x.ProjectType,
                ProjectTypeName = x.ProjectType.GetEnumName(),
                x.CarrierSize,
                x.LayersNum,
                x.CarNum,
                x.CarrierPlan,
                x.CarrierTarget,
                x.Location,
                x.Crowd,
                x.VideoUrl,
                x.UserId,
                x.UserName,
                x.UserIcon,
                x.CarrierMetroList
            });
            return ApiResult.Success(data: pageModel);
        }


        /// <summary>
        /// 获取载体实体
        /// </summary>
        /// <returns></returns>
        [HttpGet]

        [HttpPost]
        public ApiResult GetCarrierDetails()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Carrier_Extend()
            {
                ID = parms.GetInt("ID", 0),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var extModel = carrierModule.GetCarrierModel(query);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 新增/修改载体信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetSetCarrier()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            RB_Carrier_Extend query = JsonConvert.DeserializeObject<RB_Carrier_Extend>(RequestParm.msg.ToString());
            if (string.IsNullOrWhiteSpace(query.CarrierName))
            {
                return ApiResult.Failed("请输入载体名称");
            }

            if (string.IsNullOrWhiteSpace(query.Developers))
            {
                return ApiResult.Failed("请输入开发商");
            }
            if (query.BannerList == null || !query.BannerList.Any())
            {
                return ApiResult.Failed("请上传介绍图");
            }
            if (query.BannerList != null && query.BannerList.Any())
            {
                query.Banner = JsonConvert.SerializeObject(query.BannerList);
            }
            else
            {
                query.Banner = "";
            }
            if (query.HonorList != null && query.HonorList.Any())
            {
                query.Honor = JsonConvert.SerializeObject(query.HonorList);
            }
            else
            {
                query.Honor = "";
            }
            if (query.FirstShopNumList != null && query.FirstShopNumList.Any())
            {
                query.FirstShopNum = JsonConvert.SerializeObject(query.FirstShopNumList);
            }
            else
            {
                query.FirstShopNum = "";
            }
            if (query.ID == 0)
            {
                query.CreateDate = System.DateTime.Now;
                query.MallBaseId = RequestParm.MallBaseId;
                query.TenantId = RequestParm.TenantId;
            }
            query.UpdateDate = System.DateTime.Now;
            bool flag = carrierModule.SetCarrier(query);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }



        /// <summary>
        /// 删除载体信息
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveCarrier()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var CarrierId = parms.GetInt("CarrierId", 0);
            var Status = parms.GetInt("Status", 0);
            var flag = carrierModule.RemoveCarrierModule(CarrierId, RequestParm.TenantId, RequestParm.MallBaseId, Status);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }




        /// <summary>
        ///载体信息下载
        /// </summary>
        [HttpPost]
        public FileContentResult GetCarrierListToExcel()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var demodel = new RB_Carrier_Extend()
            {
                MetroName = parms.GetStringValue("MetroName"),
                CarrierName = parms.GetStringValue("CarrierName"),
                MetroNum = parms.GetInt("MetroNum", 0),
                Developers = parms.GetStringValue("Developers"),
                StartOpenTime = parms.GetStringValue("StartOpenTime"),
                EndOpenTime = parms.GetStringValue("EndOpenTime"),
                ProjectType = (Common.Enum.TradePavilion.ProjectTypeEnum)parms.GetInt("ProjectType", 0)
            };
            demodel.MallBaseId = RequestParm.MallBaseId;
            demodel.TenantId = RequestParm.TenantId;
            try
            {
                List<int> ExcelEnumIds = JsonConvert.DeserializeObject<List<int>>(parms.GetStringValue("ExcelEnumIds"));
                demodel.ExcelEnumIds = ExcelEnumIds;
                if (ExcelEnumIds == null || !ExcelEnumIds.Any())
                {
                    demodel.ExcelEnumIds = new List<int>();
                    for (int i = 1; i <= 15; i++)
                    {
                        demodel.ExcelEnumIds.Add(i);
                    }
                }
            }
            catch (Exception ex)
            {
                demodel.ExcelEnumIds = new List<int>();

                for (int i = 1; i <= 15; i++)
                {
                    demodel.ExcelEnumIds.Add(i);
                }
            }

            string ExcelName = "载体列表" + DateTime.Now.ToString("yyyyMMddHHmmss") + ".xls";
            List<ExcelDataSource> slist = new List<ExcelDataSource>();
            if (demodel.ExcelEnumIds == null || !demodel.ExcelEnumIds.Any())
            {
                var byteData1 = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData1, "application/octet-stream", ExcelName);
            }
            var Enumlist = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.CarrierExportEnum));
            ExcelDataSource header = new ExcelDataSource()
            {
                ExcelRows = new List<ExcelColumn>(30)
                {
                    new ExcelColumn(value: "序号") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER }
                }
            };
            foreach (var item in demodel.ExcelEnumIds)
            {
                var Name = Enumlist.Where(x => x.Value == item.ToString()).FirstOrDefault().Key ?? "";
                header.ExcelRows.Add(new ExcelColumn(value: Name) { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER });
            }

            slist.Add(header);

            try
            {

                var list = carrierModule.GetCarrierList(demodel);
                #region 组装数据
                int Num = 0;
                foreach (var item in list)
                {
                    Num++;
                    ExcelDataSource datarow = new ExcelDataSource()
                    {
                        ExcelRows = new List<ExcelColumn>(30) {
                            new ExcelColumn(value:Num.ToString()){ },
                        }
                    };
                    foreach (var qitem in demodel.ExcelEnumIds)
                    {
                        switch (qitem)
                        {
                            case 1:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.CarrierName) { }); break;
                            case 2:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Logo, isPic: 1) { }); break;
                            case 3:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Developers) { }); break;
                            case 4:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.VideoUrl) { }); break;
                            case 5:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Address) { }); break;
                            case 6:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.OpenTime.HasValue ? item.OpenTime.Value.ToString("yyyy-MM-dd") : "") { }); break;
                            case 7:
                                datarow.ExcelRows.Add(new ExcelColumn(value: ((item.ProjectType.HasValue && item.ProjectType.Value > 0) ? item.ProjectType.GetEnumName() : "")) { }); break;
                            case 8:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.CarrierSize.ToString()) + "万平方米") { }); break;
                            case 9:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.LayersNum) { }); break;
                            case 10:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.CarNum > 0 ? item.CarNum.ToString() : "") { }); break;
                            case 11:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.CarrierPlan) { }); break;
                            case 12:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.CarrierTarget) { }); break;
                            case 13:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Location) { }); break;
                            case 14:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Crowd) { }); break;
                            case 15:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.CarrierMetroList != null && item.CarrierMetroList.Any()) ? (string.Join(",", item.CarrierMetroList.Select(x => x.MetroNum + "号线" + x.MetroName + x.Distance))) : "") { }); break;

                        }
                    }
                    slist.Add(datarow);
                }
                #endregion
                var byteData = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData, "application/octet-stream", ExcelName);
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, string.Format("GetFileFromWebApi_requestData: {0}", JsonHelper.Serialize(RequestParm)));
                var byteData1 = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData1, "application/octet-stream", ExcelName);
            }
        }

        /// <summary>
        /// 获取载体下载枚举列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetCarrierExportEnumList()
        {
            var list = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.CarrierExportEnum));
            return ApiResult.Success("", list.Select(x => new
            {
                Name = x.Key,
                Id = Convert.ToInt32(x.Value)
            }));
        }

        #endregion


        #region 品牌管理


        /// <summary>
        /// 获取商品性质枚举
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        [AllowAnonymous]
        public ApiResult GetProjectTypeEnumList()
        {
            var list = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.ProjectTypeEnum));
            return ApiResult.Success("", list.OrderBy(x => Convert.ToInt32(x.Value)).Select(x => new
            {
                Name = x.Key,
                Id = Convert.ToInt32(x.Value)
            }));
        }

        /// <summary>
        /// 获取品牌分类分页列表
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        public ApiResult GetBrandClassPageList()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_BrandClass_Extend()
            {
                ClassName = parms.GetStringValue("ClassName")
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = carrierModule.GetBrandClassPageListRepository(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }

        /// <summary>
        /// 获取品牌分类分页列表
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        public ApiResult GetBrandClassList()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_BrandClass_Extend();
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = carrierModule.GetBrandClassList(query);
            return ApiResult.Success(data: list);
        }

        /// <summary>
        /// 获取品牌分类实体
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        public ApiResult GetBrandClassDetails()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_BrandClass_Extend()
            {
                ID = parms.GetInt("ID", 0),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var extModel = carrierModule.GetBrandClassRepository(query);
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 新增/修改品牌信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetSetBrandClass()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            RB_BrandClass_Extend query = JsonConvert.DeserializeObject<RB_BrandClass_Extend>(RequestParm.msg.ToString());
            if (string.IsNullOrWhiteSpace(query.ClassName))
            {
                return ApiResult.Failed("请输入分类名称");
            }
            //if (string.IsNullOrWhiteSpace(query.Logo))
            //{
            //    return ApiResult.Failed("请上传Logo");
            //}
            if (query.ID == 0)
            {
                query.CreateDate = System.DateTime.Now;
                query.MallBaseId = RequestParm.MallBaseId;
                query.TenantId = RequestParm.TenantId;
            }
            query.Status = 0;
            query.UpdateDate = System.DateTime.Now;
            bool flag = carrierModule.SetBrandClass(query);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 删除品牌信息
        /// </summary>
        /// <returns></returns>
        public ApiResult RemoveBrandClass()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var CarrierId = parms.GetInt("Id", 0);
            var flag = carrierModule.RemoveBrandClassModule(CarrierId);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 获取品牌分页列表
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        public ApiResult GetBrandPageList()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Brand_Extend()
            {
                BrandName = parms.GetStringValue("BrandName"),
                ProjectType = (Common.Enum.TradePavilion.ProjectTypeEnum)parms.GetInt("ProjectType", 0),
                BrandClassId = parms.GetInt("BrandClassId", 0),
                RanQi = parms.GetInt("RanQi", -1),
                KongTiao = parms.GetInt("KongTiao", -1),
                XinFeng = parms.GetInt("XinFeng", -1),
                Plumbing = parms.GetInt("Plumbing", -1),
                Caliber = parms.GetInt("Caliber", -1),
                Sewage = parms.GetInt("Sewage", -1),
                PaiYan = parms.GetInt("PaiYan", -1),
                DianLiang = parms.GetStringValue("DianLiang"),
                UserId = parms.GetInt("UserId", 0),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = carrierModule.GetBrandPageList(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query, true);
            foreach (var item in list)
            {
                item.ProjectName = item.ProjectType.GetEnumName();
                if (!string.IsNullOrWhiteSpace(item.Banner))
                {
                    item.BannerList = JsonConvert.DeserializeObject<List<string>>(item.Banner);
                }
                else
                {
                    item.BannerList = new List<string>();
                }
            }
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list;
            return ApiResult.Success(data: pageModel);
        }


        /// <summary>
        /// 获取品牌分类实体
        /// </summary>
        /// <returns></returns>
        [HttpGet]

        [HttpPost]
        public ApiResult GetBrandDetails()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Brand_Extend()
            {
                ID = parms.GetInt("ID", 0),

            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            query.RanQi = -1;
            query.KongTiao = -1;
            query.XinFeng = -1;
            query.Plumbing = -1;
            query.Caliber = -1;
            query.PaiYan = -1;
            query.Sewage = -1;
            var extModel = carrierModule.GetBrand(query);

            if (!string.IsNullOrWhiteSpace(extModel.Banner))
            {
                extModel.BannerList = JsonConvert.DeserializeObject<List<string>>(extModel.Banner);
            }
            else
            {
                extModel.BannerList = new List<string>();
            }
            return ApiResult.Success(data: extModel);
        }

        /// <summary>
        /// 新增/修改品牌信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetSetBrand()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            RB_Brand_Extend query = JsonConvert.DeserializeObject<RB_Brand_Extend>(RequestParm.msg.ToString());
            if (string.IsNullOrWhiteSpace(query.BrandName))
            {
                return ApiResult.Failed("请输入品牌名称");
            }
            if (string.IsNullOrWhiteSpace(query.Logo))
            {
                return ApiResult.Failed("请上传Logo");
            }

            if (query.BrandClassId == 0)
            {
                return ApiResult.Failed("请选择品牌分类");
            }
            if (query.BannerList != null && query.BannerList.Any())
            {
                query.Banner = JsonConvert.SerializeObject(query.BannerList);
            }
            else
            {
                query.Banner = "";
            }
            if (query.ID == 0)
            {
                query.CreateDate = System.DateTime.Now;
                query.MallBaseId = RequestParm.MallBaseId;
                query.TenantId = RequestParm.TenantId;
            }
            query.Status = 0;
            query.UpdateDate = System.DateTime.Now;
            bool flag = carrierModule.SetBrand(query);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 删除品牌信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult RemoveBrand()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var CarrierId = parms.GetInt("Id", 0);
            var flag = carrierModule.RemoveBrandModule(CarrierId);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }




        /// <summary>
        /// 品牌下载
        /// </summary>
        [HttpPost]
        public FileContentResult GetBrandListToExcel()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var demodel = new RB_Brand_Extend()
            {
                BrandName = parms.GetStringValue("BrandName"),
                ProjectType = (Common.Enum.TradePavilion.ProjectTypeEnum)parms.GetInt("ProjectType", 0),
                BrandClassId = parms.GetInt("BrandClassId", 0),
                RanQi = parms.GetInt("RanQi", -1),
                KongTiao = parms.GetInt("KongTiao", -1),
                XinFeng = parms.GetInt("XinFeng", -1),
                Plumbing = parms.GetInt("Plumbing", -1),
                Caliber = parms.GetInt("Caliber", -1),
                Sewage = parms.GetInt("Sewage", -1),
                PaiYan = parms.GetInt("PaiYan", -1),
                DianLiang = parms.GetStringValue("DianLiang"),
            };
            demodel.MallBaseId = RequestParm.MallBaseId;
            demodel.TenantId = RequestParm.TenantId;
            try
            {
                List<int> ExcelEnumIds = JsonConvert.DeserializeObject<List<int>>(parms.GetStringValue("ExcelEnumIds"));
                demodel.ExcelEnumIds = ExcelEnumIds;
                if (ExcelEnumIds == null || !ExcelEnumIds.Any())
                {
                    demodel.ExcelEnumIds = new List<int>();
                    for (int i = 1; i <= 23; i++)
                    {
                        demodel.ExcelEnumIds.Add(i);
                    }
                }
            }
            catch (Exception ex)
            {
                demodel.ExcelEnumIds = new List<int>();

                for (int i = 1; i <= 23; i++)
                {
                    demodel.ExcelEnumIds.Add(i);
                }
            }

            string ExcelName = "品牌列表" + DateTime.Now.ToString("yyyyMMddHHmmss") + ".xls";
            List<ExcelDataSource> slist = new List<ExcelDataSource>();
            if (demodel.ExcelEnumIds == null || !demodel.ExcelEnumIds.Any())
            {
                var byteData1 = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData1, "application/octet-stream", ExcelName);
            }
            var Enumlist = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.BusinessExportEnum));
            ExcelDataSource header = new ExcelDataSource()
            {
                ExcelRows = new List<ExcelColumn>(30)
                {
                    new ExcelColumn(value: "序号") { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER }
                }
            };
            foreach (var item in demodel.ExcelEnumIds)
            {
                var Name = Enumlist.Where(x => x.Value == item.ToString()).FirstOrDefault().Key ?? "";
                header.ExcelRows.Add(new ExcelColumn(value: Name) { CellWidth = 15, HAlignmentEnum = HAlignmentEnum.CENTER, VAlignmentEnum = VAlignmentEnum.CENTER });
            }

            slist.Add(header);

            try
            {

                var list = carrierModule.GetBrandListByWhere(demodel);
                #region 组装数据
                int Num = 0;
                foreach (var item in list)
                {
                    Num++;
                    ExcelDataSource datarow = new ExcelDataSource()
                    {
                        ExcelRows = new List<ExcelColumn>(30) {
                            new ExcelColumn(value:Num.ToString()){ },
                        }
                    };
                    foreach (var qitem in demodel.ExcelEnumIds)
                    {
                        switch (qitem)
                        {
                            case 1:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.ClassName) { }); break;
                            case 2:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.FullBrandName) { }); break;
                            case 4:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Logo, isPic: 1) { }); break;
                            case 3:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.BrandName) { }); break;
                            case 5:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.VideoUrl) { }); break;
                            case 6:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.ShopNum.ToString()) { }); break;
                            case 7:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.BuiltUpArea.ToString()) + "-" + (item.EndBuiltUpArea.ToString())) { }); break;
                            case 8:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.AreaRequirement.ToString()) + "-" + (item.EndAreaRequirement.ToString())) { }); break;
                            case 9:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.CustomerType) { }); break;
                            case 10:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.BrandType) { }); break;
                            case 11:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.StoreExpansion) { }); break;
                            case 12:
                                datarow.ExcelRows.Add(new ExcelColumn(value: ((item.ProjectType.HasValue && item.ProjectType.Value > 0) ? item.ProjectType.GetEnumName() : "")) { }); break;
                            case 13:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.IsShopSize == 1 ? ("有" + (!string.IsNullOrWhiteSpace(item.ShopSize) ? item.ShopSize : "")) : "无")) { }); break;
                            case 14:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.RanQi == 1 ? ("有" + (!string.IsNullOrWhiteSpace(item.RanQiBZ) ? item.RanQiBZ : "")) : "无")) { }); break;
                            case 15:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.PaiYan == 1 ? ("有" + (!string.IsNullOrWhiteSpace(item.PaiYanBZ) ? item.PaiYanBZ : "")) : "无")) { }); break;
                            case 16:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.IsDianLiang == 1 ? ("有" + (!string.IsNullOrWhiteSpace(item.DianLiang) ? item.DianLiang : "")) : "无")) { }); break;
                            case 17:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.Plumbing == 1 ? "有" : "无")) { }); break;
                            case 18:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.Caliber == 1 ? ("有") : "无")) { }); break;
                            case 19:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.Sewage == 1 ? ("有") : "无")) { }); break;
                            case 20:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.KongTiao == 1 ? ("有") : "无")) { }); break;
                            case 21:
                                datarow.ExcelRows.Add(new ExcelColumn(value: (item.XinFeng == 1 ? ("有") : "无")) { }); break;
                            case 22:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Advertising) { }); break;
                            case 23:
                                datarow.ExcelRows.Add(new ExcelColumn(value: item.Other) { }); break;
                        }
                    }
                    slist.Add(datarow);
                }
                #endregion
                var byteData = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData, "application/octet-stream", ExcelName);
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, string.Format("GetFileFromWebApi_requestData: {0}", JsonHelper.Serialize(RequestParm)));
                var byteData1 = ExcelTempLateHelper.ToExcelExtend(slist);
                return File(byteData1, "application/octet-stream", ExcelName);
            }
        }

        /// <summary>
        /// 获取品牌下载枚举列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetBusinessExportEnumList()
        {
            var list = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.BusinessExportEnum));
            return ApiResult.Success("", list.Select(x => new
            {
                Name = x.Key,
                Id = Convert.ToInt32(x.Value)
            }));
        }

        #endregion

        #region 榜单管理

        /// <summary>
        /// 获取榜单分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetFirstShopPrizeListPageList()
        {
            var req = base.RequestParm;
            ResultPageModel pageModel = JsonHelper.DeserializeObject<ResultPageModel>(req.msg.ToString());
            RB_FirstShop_List_Extend demodel = JsonHelper.DeserializeObject<RB_FirstShop_List_Extend>(req.msg.ToString());
            demodel.TenantId = req.TenantId;
            demodel.MallBaseId = req.MallBaseId;

            var list = firstShopListModule.GetFirstShopListPageList(pageModel.pageIndex, pageModel.pageSize, out long count, demodel);
            return ApiResult.Success("", list.Select(x => new
            {
                x.Id,
                x.ListType,
                ListTypeName = x.ListType.GetEnumName(),
                x.ObjectTypeIds,
                x.ObjectTypeList,
                x.Sponsor,
                x.ListName,
                x.CoverImage,
                EndDate = x.EndDate.ToString("yyyy-MM-dd"),
                StartDate = x.StartDate.ToString("yyyy-MM-dd"),
                FinishDate = x.FinishDate.ToString("yyyy-MM-dd HH:mm"),
                x.Content,
                x.EnableRank,
                x.PrizeNum,
                x.ListState,
                ListStateName = x.ListState.GetEnumName(),
                UpdateDate = x.UpdateDate.ToString("yyyy-MM-dd HH:mm:ss")
            }));
        }

        /// <summary>
        /// 获取榜单详情
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetFistShopPrizeListInfo() {
            var req = base.RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int ListId = parms.GetInt("ListId", 0);//榜单id
            if (ListId <= 0) {
                return ApiResult.ParamIsNull();
            }

            var model = firstShopListModule.GetFirstShopListInfo(ListId);
            if (model == null) { return ApiResult.Failed("榜单不存在"); }
            return ApiResult.Success("", new
            {
                model.Id,
                model.ListType,
                model.ObjectTypeIds,                
                model.Sponsor,
                model.ListName,
                model.CoverImage,
                EndDate = model.EndDate.ToString("yyyy-MM-dd"),
                StartDate = model.StartDate.ToString("yyyy-MM-dd"),
                FinishDate = model.FinishDate.ToString("yyyy-MM-dd HH:mm"),
                model.Content,
                model.EnableRank,
                model.PrizeNum,
                model.ListState
            });
        }

        /// <summary>
        /// 设置榜单
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult SetFistShopPrizeListInfo() {
            var req = base.RequestParm;
            RB_FirstShop_List_Extend demodel = JsonHelper.DeserializeObject<RB_FirstShop_List_Extend>(req.msg.ToString());
            if (string.IsNullOrEmpty(demodel.Sponsor)) {
                return ApiResult.ParamIsNull("请输入主办方名称");
            }
            if (string.IsNullOrEmpty(demodel.ListName)) {
                return ApiResult.ParamIsNull("请输入榜单名称");
            }
            if (string.IsNullOrEmpty(demodel.ObjectTypeIds)) {
                return ApiResult.ParamIsNull("请选择面向群体");
            }
            if (string.IsNullOrEmpty(demodel.CoverImage)) {
                return ApiResult.ParamIsNull("请上传封面图");
            }
            if (string.IsNullOrEmpty(demodel.Content)) {
                return ApiResult.ParamIsNull("请输入详情内容");
            }
            if (demodel.PrizeNum <= 0) {
                return ApiResult.ParamIsNull("请输入入围名额");
            }

            demodel.EndDate = Convert.ToDateTime(demodel.EndDate.ToString("yyyy-MM-dd") + " 23:59:59");
            demodel.ListState = Common.Enum.TradePavilion.ListStateEnum.Enroll;
            demodel.TenantId = req.TenantId;
            demodel.MallBaseId = req.MallBaseId;
            demodel.Status = 0;
            demodel.CreateBy = req.EmpId;
            demodel.CreateDate = DateTime.Now;
            demodel.UpdateBy = req.EmpId;
            demodel.UpdateDate = DateTime.Now;

            string msg = firstShopListModule.SetFirstShopListInfo(demodel);
            if (msg == "")
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed(msg);
            }
        }

        /// <summary>
        /// 取消榜单
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelFistShopPrizeListInfo() {
            var req = base.RequestParm;
            JObject parms = JObject.Parse(req.msg.ToString());
            int ListId = parms.GetInt("ListId", 0);//榜单id
            if (ListId <= 0)
            {
                return ApiResult.ParamIsNull();
            }

            string msg = firstShopListModule.DelFirstShopListInfo(ListId, req.MallBaseId);
            if (msg == "")
            {
                return ApiResult.Success();
            }
            else {
                return ApiResult.Failed(msg);
            }
        }

        /// <summary>
        /// 获取榜单面向群体枚举列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetListObjectTypeEnumList()
        {
            var list = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.ListObjectTypeEnum));
            return ApiResult.Success("", list.Select(x => new
            {
                Name = x.Key,
                Id = Convert.ToInt32(x.Value)
            }));
        }

        /// <summary>
        /// 获取榜单状态枚举列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetListStateEnumList()
        {
            var list = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.ListStateEnum));
            return ApiResult.Success("", list.Select(x => new
            {
                Name = x.Key,
                Id = Convert.ToInt32(x.Value)
            }));
        }

        /// <summary>
        /// 获取榜单类型枚举列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetListTypeEnumList()
        {
            var list = EnumHelper.GetEnumList(typeof(Common.Enum.TradePavilion.ListTypeEnum));
            return ApiResult.Success("", list.Select(x => new
            {
                Name = x.Key,
                Id = Convert.ToInt32(x.Value)
            }));
        }
        #endregion

        #region 心愿单列表

        /// <summary>
        /// 获取心愿单分页列表
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetFirstShopWishPageList()
        {
            var requst = RequestParm;
            ResultPageModel pageModel = JsonHelper.DeserializeObject<ResultPageModel>(requst.msg.ToString());
            RB_FirstShop_Wish_Extend demodel = JsonHelper.DeserializeObject<RB_FirstShop_Wish_Extend>(requst.msg.ToString());
            demodel.MallBaseId = requst.MallBaseId;
            demodel.TenantId = requst.TenantId;

            var list = firstShopWishModule.GetWishPageList(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, demodel);
            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list.Select(x => new
            {
                x.Id,
                x.UserId,
                x.UserName,
                x.BranchName,
                x.CarrierName,
                x.Area,
                x.Description,
                x.NameCard,
                x.Status,
                UpdateDate = x.UpdateDate.ToString("yyyy-MM-dd HH:mm")
            });
            return ApiResult.Success("", pageModel);
        }

        /// <summary>
        /// 取消心愿单
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult DelFirstShopWishInfo()
        {
            var requst = RequestParm;
            JObject prams = JObject.Parse(requst.msg.ToString());
            int WishId = prams.GetInt("WishId", 0);
            if (WishId <= 0)
            {
                return ApiResult.ParamIsNull();
            }

            string msg = firstShopWishModule.DelFirstShopWishInfo(WishId, requst.MallBaseId);
            if (msg == "")
            {
                return ApiResult.Success();
            }
            else
            {
                return ApiResult.Failed();
            }
        }

        #endregion

        #region 奖项
        /// <summary>
        /// 获取奖项分页列表
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        [HttpPost]
        public ApiResult GetPrizePageList()
        {
            ResultPageModel pageModel = JsonConvert.DeserializeObject<ResultPageModel>(RequestParm.msg.ToString());
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Prize_Extend()
            {
                Name = parms.GetStringValue("Name"),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var list = carrierModule.GetPrizePageList(pageModel.pageIndex, pageModel.pageSize, out long rowsCount, query);

            pageModel.count = Convert.ToInt32(rowsCount);
            pageModel.pageData = list.Select(x => new
            {
                x.Id,
                x.Name,
                x.BigIco,
                x.SmallIco,
                x.TotalCount,
                x.YearInfo
            });
            return ApiResult.Success(data: pageModel);
        }


        /// <summary>
        /// 获取奖项实体
        /// </summary>
        /// <returns></returns>
        [HttpGet]

        [HttpPost]
        public ApiResult GetPrizeDetails()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Prize_Extend()
            {
                Id = parms.GetInt("Id", 0),
            };
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var extModel = carrierModule.GetPrizeList(query).FirstOrDefault();
            if (extModel == null || extModel.Id == 0)
            {
                return ApiResult.Failed("奖项信息不存在");
            }
            return ApiResult.Success(data: extModel);
        }



        /// <summary>
        /// 获取奖项实体
        /// </summary>
        /// <returns></returns>
        [HttpGet]

        [HttpPost]
        public ApiResult GetPrizeList()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var query = new RB_Prize_Extend();
            query.MallBaseId = RequestParm.MallBaseId;
            query.TenantId = RequestParm.TenantId;
            var extList = carrierModule.GetPrizeList(query);
            return ApiResult.Success(data: extList);
        }



        /// <summary>
        /// 新增/修改奖项信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult GetSetPrize()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            RB_Prize_Extend query = JsonConvert.DeserializeObject<RB_Prize_Extend>(RequestParm.msg.ToString());
            if (string.IsNullOrWhiteSpace(query.Name))
            {
                return ApiResult.Failed("请输入奖项名称");
            }
            if (string.IsNullOrWhiteSpace(query.BigIco))
            {
                return ApiResult.Failed("请上传大奖章");
            }
            if (string.IsNullOrWhiteSpace(query.SmallIco))
            {
                return ApiResult.Failed("请上传小奖章");
            }

            if (query.Id == 0)
            {
                query.CreateDate = System.DateTime.Now;
                query.MallBaseId = RequestParm.MallBaseId;
                query.TenantId = RequestParm.TenantId;
            }
            query.Status = 0;
            bool flag = carrierModule.SetPrize(query);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }

        /// <summary>
        /// 删除奖项信息
        /// </summary>
        /// <returns></returns>
        [HttpPost]
        public ApiResult RemovePrize()
        {
            JObject parms = JObject.Parse(RequestParm.msg.ToString());
            var Id = parms.GetInt("Id", 0);
            var flag = carrierModule.RemovePrizeModule(Id);
            return flag ? ApiResult.Success() : ApiResult.Failed();
        }


        #endregion
    }
}
