﻿using ICSharpCode.SharpZipLib.Checksum;
using ICSharpCode.SharpZipLib.Zip;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace Mall.Common.Plugin
{
    /// <summary>
    /// 枚举帮助类
    /// </summary>
    public class ZipHelper
    {
        #region zip压缩
        /// <summary>   
        /// 压缩文件或文件夹 ----无密码 
        /// </summary>   
        /// <param name="fileToZip">要压缩的路径-文件夹或者文件</param>   
        /// <param name="zipedFile">压缩后的文件名</param>
        /// <param name="errorOut">如果失败返回失败信息</param>
        /// <returns>压缩结果</returns>   
        public static bool Zip(string fileToZip, string zipedFile, ref string errorOut)
        {
            bool result = false;
            try
            {
                if (Directory.Exists(fileToZip))
                    result = ZipDirectory(fileToZip, zipedFile, null);
                else if (File.Exists(fileToZip))
                    result = ZipFile(fileToZip, zipedFile, null);
            }
            catch (Exception ex)
            {
                LogHelper.Write(ex, "Zip");
                errorOut = ex.Message;
            }
            return result;
        }
        /// <summary>   
        /// 压缩文件   
        /// </summary>   
        /// <param name="fileToZip">要压缩的文件全名</param>   
        /// <param name="zipedFile">压缩后的文件名</param>   
        /// <param name="password">密码</param>   
        /// <returns>压缩结果</returns>   
        private static bool ZipFile(string fileToZip, string zipedFile, string password)
        {
            bool result = true;
            ZipOutputStream zipStream = null;
            FileStream fs = null;
            ZipEntry ent = null;

            if (!File.Exists(fileToZip))
                return false;

            try
            {
                fs = File.OpenRead(fileToZip);
                byte[] buffer = new byte[fs.Length];
                fs.Read(buffer, 0, buffer.Length);
                fs.Close();

                fs = File.Create(zipedFile);
                zipStream = new ZipOutputStream(fs);
                if (!string.IsNullOrEmpty(password)) zipStream.Password = password;
                ent = new ZipEntry(Path.GetFileName(fileToZip));
                zipStream.PutNextEntry(ent);
                zipStream.SetLevel(6);

                zipStream.Write(buffer, 0, buffer.Length);

            }
            catch (Exception ex)
            {
                result = false;
                throw ex;
            }
            finally
            {
                if (zipStream != null)
                {
                    zipStream.Finish();
                    zipStream.Close();
                }
                if (ent != null)
                {
                    ent = null;
                }
                if (fs != null)
                {
                    fs.Close();
                    fs.Dispose();
                }
            }
            GC.Collect();
            GC.Collect(1);

            return result;
        }

        /// <summary>
        /// 压缩文件夹
        /// </summary>
        /// <param name="strFile">带压缩的文件夹目录</param>
        /// <param name="strZip">压缩后的文件名</param>
        /// <param name="password">压缩密码</param>
        /// <returns>是否压缩成功</returns>
        private static bool ZipDirectory(string strFile, string strZip, string password)
        {
            bool result = false;
            if (!Directory.Exists(strFile)) return false;
            if (strFile[strFile.Length - 1] != Path.DirectorySeparatorChar)
                strFile += Path.DirectorySeparatorChar;
            ZipOutputStream s = new ZipOutputStream(File.Create(strZip));
            s.SetLevel(6); // 0 - store only to 9 - means best compression
            if (!string.IsNullOrEmpty(password)) s.Password = password;
            try
            {
                result = Zip(strFile, s, strFile);
            }
            catch (Exception ex)
            {
                throw ex;
            }
            finally
            {
                s.Finish();
                s.Close();
            }
            return result;
        }
        /// <summary>
        /// 压缩文件夹内部方法
        /// </summary>
        /// <param name="strFile"></param>
        /// <param name="s"></param>
        /// <param name="staticFile"></param>
        /// <returns></returns>
        private static bool Zip(string strFile, ZipOutputStream s, string staticFile)
        {
            bool result = true;
            if (strFile[strFile.Length - 1] != Path.DirectorySeparatorChar) strFile += Path.DirectorySeparatorChar;
            Crc32 crc = new Crc32();
            try
            {
                string[] filenames = Directory.GetFileSystemEntries(strFile);
                foreach (string file in filenames)
                {

                    if (Directory.Exists(file))
                    {
                        Zip(file, s, staticFile);
                    }

                    else // 否则直接压缩文件
                    {
                        //打开压缩文件
                        FileStream fs = File.OpenRead(file);

                        byte[] buffer = new byte[fs.Length];
                        fs.Read(buffer, 0, buffer.Length);
                        string tempfile = file.Substring(staticFile.LastIndexOf("\\") + 1);
                        ZipEntry entry = new ZipEntry(tempfile)
                        {
                            DateTime = DateTime.Now,
                            Size = fs.Length
                        };
                        fs.Close();
                        crc.Reset();
                        crc.Update(buffer);
                        entry.Crc = crc.Value;
                        s.PutNextEntry(entry);

                        s.Write(buffer, 0, buffer.Length);
                    }
                }
            }
            catch (Exception ex)
            {
                result = false;
                throw ex;
            }
            return result;

        }
        #endregion
    }
}
