﻿using Mall.DataAccess.Utils;
using System;
using System.Data;

namespace Mall.DataAccess
{

    /// <summary>
    /// 数据库接口
    /// </summary>
    public interface IDatabase
    {
        /// <summary>
        /// 数据库连接对象
        /// </summary>
        IDbConnection Connection { get; }

        /// <summary>
        /// 数据库类型DataBaseType枚举
        /// </summary>
        DataBaseType DbType { get; }

        /// <summary>
        /// 数据库连接字符串
        /// </summary>
        string ConnKey { get; }
    }

    /// <summary>
    /// 数据库类对象
    /// </summary>
    public class Database : IDatabase
    {
        /// <summary>
        /// 数据库连接对象
        /// </summary>
        public IDbConnection Connection { get; private set; }

        /// <summary>
        /// 数据库类型DataBaseType枚举
        /// </summary>
        public DataBaseType DbType { get; private set; }

        /// <summary>
        /// 数据库连接字符串
        /// </summary>
        public string ConnKey { get; set; }

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="connKey">连接字符串</param>
        public Database(string connKey)
        {
            this.ConnKey = connKey;
            Connection = DBUtils.CreateDBConnection(connKey, out DataBaseType dbType);
            this.DbType = dbType;
        }

        /// <summary>
        /// 数据库连接字符串
        /// </summary>
        /// <param name="dbType">数据库类型DataBaseType枚举</param>
        /// <param name="connKey">数据库连接字符串</param>
        public Database(DataBaseType dbType, string connKey)
        {
            this.ConnKey = connKey;
            this.DbType = dbType;
            this.Connection = DBUtils.CreateDBConnection(dbType, connKey);
        }
    }


    /// <summary>
    /// 数据连接事务的Session接口
    /// </summary>
    public interface IDBSession : IDisposable
    {
        /// <summary>
        /// 连接字符串
        /// </summary>
        string ConnKey { get; }

        /// <summary>
        /// 数据库类型DataBaseType枚举
        /// </summary>
        DataBaseType DbType { get; }

        /// <summary>
        /// 数据库连接对象
        /// </summary>
        IDbConnection Connection { get; }

        /// <summary>
        /// 数据库事务
        /// </summary>
        IDbTransaction Transaction { get; set; }

        /// <summary>
        /// 开始数据库事务
        /// </summary>
        /// <param name="isolation">连接的事务锁定行为(默认:正在读取数据以避免脏读，但在该事务，从而导致不可重复读取或幻像数据末尾之前可以更改数据时，将保持共享的锁。)</param>
        /// <returns></returns>
        IDbTransaction Begin(IsolationLevel isolation = IsolationLevel.ReadCommitted);

        /// <summary>
        /// 提交事务
        /// </summary>
        void Commit();

        /// <summary>
        /// 提交事务
        /// </summary>
        /// <param name="methodName">方法名称</param>
        void Commit(string methodName);

        /// <summary>
        /// 回滚事务
        /// </summary>
        void Rollback();

        /// <summary>
        /// 回滚事务扩展
        /// </summary>
        /// <param name="methodName">方法名称</param>
        void Rollback(string methodName);
    }
}
