var lunarCalendar = {
    /**
      * 农历1900-2100的润大小信息表
      * @Array Of Property
      * @return Hex 
      */
    lunarInfo: [0x04bd8, 0x04ae0, 0x0a570, 0x054d5, 0x0d260, 0x0d950, 0x16554, 0x056a0, 0x09ad0, 0x055d2,//1900-1909
        0x04ae0, 0x0a5b6, 0x0a4d0, 0x0d250, 0x1d255, 0x0b540, 0x0d6a0, 0x0ada2, 0x095b0, 0x14977,//1910-1919
        0x04970, 0x0a4b0, 0x0b4b5, 0x06a50, 0x06d40, 0x1ab54, 0x02b60, 0x09570, 0x052f2, 0x04970,//1920-1929
        0x06566, 0x0d4a0, 0x0ea50, 0x06e95, 0x05ad0, 0x02b60, 0x186e3, 0x092e0, 0x1c8d7, 0x0c950,//1930-1939
        0x0d4a0, 0x1d8a6, 0x0b550, 0x056a0, 0x1a5b4, 0x025d0, 0x092d0, 0x0d2b2, 0x0a950, 0x0b557,//1940-1949
        0x06ca0, 0x0b550, 0x15355, 0x04da0, 0x0a5b0, 0x14573, 0x052b0, 0x0a9a8, 0x0e950, 0x06aa0,//1950-1959
        0x0aea6, 0x0ab50, 0x04b60, 0x0aae4, 0x0a570, 0x05260, 0x0f263, 0x0d950, 0x05b57, 0x056a0,//1960-1969
        0x096d0, 0x04dd5, 0x04ad0, 0x0a4d0, 0x0d4d4, 0x0d250, 0x0d558, 0x0b540, 0x0b6a0, 0x195a6,//1970-1979
        0x095b0, 0x049b0, 0x0a974, 0x0a4b0, 0x0b27a, 0x06a50, 0x06d40, 0x0af46, 0x0ab60, 0x09570,//1980-1989
        0x04af5, 0x04970, 0x064b0, 0x074a3, 0x0ea50, 0x06b58, 0x055c0, 0x0ab60, 0x096d5, 0x092e0,//1990-1999
        0x0c960, 0x0d954, 0x0d4a0, 0x0da50, 0x07552, 0x056a0, 0x0abb7, 0x025d0, 0x092d0, 0x0cab5,//2000-2009
        0x0a950, 0x0b4a0, 0x0baa4, 0x0ad50, 0x055d9, 0x04ba0, 0x0a5b0, 0x15176, 0x052b0, 0x0a930,//2010-2019
        0x07954, 0x06aa0, 0x0ad50, 0x05b52, 0x04b60, 0x0a6e6, 0x0a4e0, 0x0d260, 0x0ea65, 0x0d530,//2020-2029
        0x05aa0, 0x076a3, 0x096d0, 0x04afb, 0x04ad0, 0x0a4d0, 0x1d0b6, 0x0d250, 0x0d520, 0x0dd45,//2030-2039
        0x0b5a0, 0x056d0, 0x055b2, 0x049b0, 0x0a577, 0x0a4b0, 0x0aa50, 0x1b255, 0x06d20, 0x0ada0,//2040-2049
        0x14b63, 0x09370, 0x049f8, 0x04970, 0x064b0, 0x168a6, 0x0ea50, 0x06b20, 0x1a6c4, 0x0aae0,//2050-2059
        0x0a2e0, 0x0d2e3, 0x0c960, 0x0d557, 0x0d4a0, 0x0da50, 0x05d55, 0x056a0, 0x0a6d0, 0x055d4,//2060-2069
        0x052d0, 0x0a9b8, 0x0a950, 0x0b4a0, 0x0b6a6, 0x0ad50, 0x055a0, 0x0aba4, 0x0a5b0, 0x052b0,//2070-2079
        0x0b273, 0x06930, 0x07337, 0x06aa0, 0x0ad50, 0x14b55, 0x04b60, 0x0a570, 0x054e4, 0x0d160,//2080-2089
        0x0e968, 0x0d520, 0x0daa0, 0x16aa6, 0x056d0, 0x04ae0, 0x0a9d4, 0x0a2d0, 0x0d150, 0x0f252,//2090-2099
        0x0d520],
    /**
      * 公历每个月份的天数普通表
      * @Array Of Property
      * @return Number 
      */
    solarMonth: [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31],
    /**
      * 数字转中文速查表
      * @Array Of Property 
      * @trans ['日','一','二','三','四','五','六','七','八','九','十']
      * @return Cn string 
      */
    weekStr: ["\u65e5", "\u4e00", "\u4e8c", "\u4e09", "\u56db", "\u4e94", "\u516d", "\u4e03", "\u516b", "\u4e5d", "\u5341"],
    weekStr2: ["SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"],
    /**
      * 日期转农历称呼速查表
      * @Array Of Property 
      * @trans ['初','十','廿','卅']
      * @return Cn string 
      */
    nStr2: ["\u521d", "\u5341", "\u5eff", "\u5345"],

    /**
      * 月份转农历称呼速查表
      * @Array Of Property 
      * @trans ['正','一','二','三','四','五','六','七','八','九','十','冬','腊']
      * @return Cn string 
      */
    nStr3: ["\u6b63", "\u4e8c", "\u4e09", "\u56db", "\u4e94", "\u516d", "\u4e03", "\u516b", "\u4e5d", "\u5341", "\u51ac", "\u814a"],

    /**
      * 返回农历y年一整年的总天数
      * @param lunar Year
      * @return Number
      * @eg:var count = lunarCalendar.lYearDays(1987) ;//count=387
      */
    lYearDays: function (y) {
        var i, sum = 348;
        for (i = 0x8000; i > 0x8; i >>= 1) { sum += (lunarCalendar.lunarInfo[y - 1900] & i) ? 1 : 0; }
        return (sum + lunarCalendar.leapDays(y));
    },

    /**
      * 返回农历y年闰月是哪个月；若y年没有闰月 则返回0
      * @param lunar Year
      * @return Number (0-12)
      * @eg:var leapMonth = lunarCalendar.leapMonth(1987) ;//leapMonth=6
      */
    leapMonth: function (y) { //闰字编码 \u95f0
        return (lunarCalendar.lunarInfo[y - 1900] & 0xf);
    },

    /**
      * 返回农历y年闰月的天数 若该年没有闰月则返回0
      * @param lunar Year
      * @return Number (0、29、30)
      * @eg:var leapMonthDay = lunarCalendar.leapDays(1987) ;//leapMonthDay=29
      */
    leapDays: function (y) {
        if (lunarCalendar.leapMonth(y)) {
            return ((lunarCalendar.lunarInfo[y - 1900] & 0x10000) ? 30 : 29);
        }
        return (0);
    },
    /**
      * 返回农历y年m月（非闰月）的总天数，计算m为闰月时的天数请使用leapDays方法
      * @param lunar Year
      * @return Number (-1、29、30)
      * @eg:var MonthDay = lunarCalendar.monthDays(1987,9) ;//MonthDay=29
      */
    monthDays: function (y, m) {
        if (m > 12 || m < 1) { return -1 }//月份参数从1至12，参数错误返回-1
        return ((lunarCalendar.lunarInfo[y - 1900] & (0x10000 >> m)) ? 30 : 29);
    },

    /**
      * 返回公历(!)y年m月的天数
      * @param solar Year
      * @return Number (-1、28、29、30、31)
      * @eg:var solarMonthDay = lunarCalendar.leapDays(1987) ;//solarMonthDay=30
      */
    solarDays: function (y, m) {
        if (m > 12 || m < 1) { return -1 } //若参数错误 返回-1
        var ms = m - 1;
        if (ms == 1) { //2月份的闰平规律测算后确认返回28或29
            return (((y % 4 == 0) && (y % 100 != 0) || (y % 400 == 0)) ? 29 : 28);
        } else {
            return (lunarCalendar.solarMonth[ms]);
        }
    },

    /**
      * 传入农历数字月份返回汉语通俗表示法
      * @param lunar month
      * @return Cn string
      * @eg:var cnMonth = lunarCalendar.toChinaMonth(12) ;//cnMonth='腊月'
      */
    toChinaMonth: function (m) { // 月 => \u6708
        if (m > 12 || m < 1) { return -1 } //若参数错误 返回-1
        var s = lunarCalendar.nStr3[m - 1];
        s += "\u6708";//加上月字
        return s;
    },

    /**
      * 传入农历日期数字返回汉字表示法
      * @param lunar day
      * @return Cn string
      * @eg:var cnDay = lunarCalendar.toChinaDay(21) ;//cnMonth='廿一'
      */
    toChinaDay: function (d) { //日 => \u65e5
        var s;
        switch (d) {
            case 10:
                s = '\u521d\u5341'; break;
            case 20:
                s = '\u4e8c\u5341'; break;
                break;
            case 30:
                s = '\u4e09\u5341'; break;
                break;
            default:
                s = lunarCalendar.nStr2[Math.floor(d / 10)];
                s += lunarCalendar.weekStr[d % 10];
        }
        return (s);
    },
    isLeap: function (objDate) {
        var i, leap = 0, temp = 0;
        var offset = (Date.UTC(objDate.getFullYear(), objDate.getMonth(), objDate.getDate()) - Date.UTC(1900, 0, 31)) / 86400000;
        for (i = 1900; i < 2101 && offset > 0; i++) { temp = lunarCalendar.lYearDays(i); offset -= temp; }
        if (offset < 0) { offset += temp; i--; }
        //农历年
        var year = i;
        var leap = lunarCalendar.leapMonth(i); //闰哪个月
        var isLeap = false;
        //效验闰月
        for (i = 1; i < 13 && offset > 0; i++) {
            //闰月
            if (leap > 0 && i == (leap + 1) && isLeap == false) {
                --i;
                isLeap = true;
                temp = lunarCalendar.leapDays(year); //计算农历闰月天数
            }
            else {
                temp = lunarCalendar.monthDays(year, i);//计算农历普通月天数
            }
            //解除闰月
            if (isLeap == true && i == (leap + 1)) { isLeap = false; }
            offset -= temp;
        }
        if (offset == 0 && leap > 0 && i == leap + 1)
            if (isLeap) {
                isLeap = false;
            } else {
                isLeap = true;
                --i;
            }
        if (offset < 0) { offset += temp; --i; }
        //农历月
        var lunarmonth = i;
        //农历日
        var lunarDay = offset + 1;
        return { isLeap, year, lunarmonth, lunarDay };
    },
    /**
      * 传入阳历年月日获得详细的公历、农历object信息  lunarCalendar.initLunar(1987,11)<=>Array
      * @param y  solar year
      * @param m  solar month
      * @return Array
      */
    initLunar: function (yearStr, monthStr, dayNum, dateStr) {
        //年份限定、上限
        if (yearStr < 1900 || yearStr > 2100) {
            return -1;
        }
        var objDate = new Date();
        var endDate = new Date();
        if (dateStr) {
            let d = Date.parse(dateStr)
            let res = new Date(d);

            objDate = res
            let d2 = Date.parse(dateStr)
            let res2 = new Date(d2);
            endDate = res2
        }


        endDate.setDate(endDate.getDate() + dayNum);
        //修正ymd参数
        var m = objDate.getMonth() + 1, d = objDate.getDate();
        var totalMonthDay = lunarCalendar.solarDays(yearStr, m);
        var array = [];

        while ((endDate.getTime() - objDate.getTime()) > 0) {
            //星期几
            var nWeek = objDate.getDay(), cWeek = lunarCalendar.weekStr[nWeek];
            var leepObj = lunarCalendar.isLeap(objDate);
            var obj = {
                CurrentYear: objDate.getFullYear(), //当前年
                CurrentMonth: objDate.getMonth() + 1,//当前月
                CurrentDay: objDate.getDate(),//当前天
                CurrentMonthCn: objDate.getMonth() + 1,//农历月中文
                CurrentDate: objDate.getFullYear() + "-" + (objDate.getMonth() + 1) + "-" + objDate.getDate(),
                Week: cWeek,//星期
                LunarYear: leepObj.year,//农历年
                LunarMonth: leepObj.lunarmonth,//农历月
                LunarMonthCn: (leepObj.isLeap ? "\u95f0" : '') + lunarCalendar.toChinaMonth(leepObj.lunarmonth),//农历月中文
                LunarDay: leepObj.lunarDay,//农历日
                LunarDayCn: lunarCalendar.toChinaDay(leepObj.lunarDay),//农历日中文
                IsLeap: leepObj.isLeap,//是否润月
            };
            array.push(obj);
            objDate.setDate(objDate.getDate() + 1);
        }
        return array;
    },
    changeLunar: function (dateStr, dayNum) {
        var objDate = new Date(dateStr);
        objDate.setDate(objDate.getDate() + (dayNum * 1));
        //星期几
        var nWeek = objDate.getDay(), cWeek = lunarCalendar.weekStr[nWeek];
        var leepObj = lunarCalendar.isLeap(objDate);
        var obj = {
            CurrentYear: objDate.getFullYear(), //当前年
            CurrentMonth: objDate.getMonth() + 1,//当前月
            CurrentDay: objDate.getDate(),//当前天
            CurrentMonthCn: objDate.getMonth() + 1,//农历月中文
            CurrentDate: objDate.getFullYear() + "-" + (objDate.getMonth() + 1) + "-" + objDate.getDate(),
            Week: cWeek,//星期
            LunarYear: leepObj.year,//农历年
            LunarMonth: leepObj.lunarmonth,//农历月
            LunarMonthCn: (leepObj.isLeap ? "\u95f0" : '') + lunarCalendar.toChinaMonth(leepObj.lunarmonth),//农历月中文
            LunarDay: leepObj.lunarDay,//农历日
            LunarDayCn: lunarCalendar.toChinaDay(leepObj.lunarDay),//农历日中文
            IsLeap: leepObj.isLeap,//是否润月
        };
        return obj;
    }
};
export default lunarCalendar;