﻿using JWT.Exceptions;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Http.Features;
using Microsoft.AspNetCore.Mvc.Filters;
using Newtonsoft.Json.Linq;
using System;
using System.Net;
using VTX.FW.Api;
using VTX.FW.Filter;
using VTX.FW.Helper;

namespace Edu.WebApi.Filter
{
    /// <summary>
    /// Api过滤属性
    /// </summary>
    public class ActionFilterAttribute : ApiFilterAttribute
    {
        /// <summary>
        /// token验证
        /// </summary>
        /// <param name="actionContext"></param>
        public override void OnActionExecuting(ActionExecutingContext actionContext)
        {
            base.OnActionExecuting(actionContext);
            HttpRequest request = actionContext.HttpContext.Request;
            request.EnableBuffering();
            RequestParm requestParm = new()
            {
                Token = request.Headers["Authorization"]
            };
            bool isCheckToken = true;
            var endpoint = actionContext.HttpContext.Features.Get<IEndpointFeature>()?.Endpoint;
            if (endpoint != null && endpoint.Metadata.GetMetadata<AllowAnonymousAttribute>() != null)
            {
                isCheckToken = false;
            }
            //Token校验
            if (isCheckToken)
            {
                string token = (requestParm?.Token ?? "").Replace("Bearer", "").Trim();
                JWTValidat(actionContext, token);
            }
        }

        /// <summary>
        /// token校验
        /// </summary>
        /// <param name="actionContext"></param>
        /// <param name="token"></param>
        private static void JWTValidat(ActionExecutingContext actionContext, string token)
        {
            if (!string.IsNullOrEmpty(token))
            {
                //解析token，校验是否失效
                try
                {
                    JObject jwtJson = VTX.FW.Helper.JwtHelper.AnalysisToken(token);
                    var BaseUserId = jwtJson.GetInt("BaseUserId");
                    if (BaseUserId > 0)
                    {
                        var cacheUser = EduSpider.Cache.User.UserReidsCache.GetUserLoginInfo(BaseUserId);
                        if ((cacheUser.Token != token || string.IsNullOrWhiteSpace(cacheUser.Token)))
                        {
                            actionContext.Result = new Microsoft.AspNetCore.Mvc.JsonResult(
                                new ApiResult
                                {
                                    Code = (int)ResultCode.TokenOverdue,
                                    Message = "用户凭证失效，请重新登录0",
                                    Data = ""
                                });
                        }
                    }
                }
                catch (SignatureVerificationException sve)
                {
                    string message = sve.Message;
                    actionContext.Result = new Microsoft.AspNetCore.Mvc.JsonResult(
                         new ApiResult
                         {
                             Code = (int)ResultCode.TokenOverdue,
                             Message = "用户凭证失效，请重新登录1",
                             Data = ""
                         });
                }
                catch (ArgumentException ae)
                {
                    string message = ae.Message;
                    actionContext.Result = new Microsoft.AspNetCore.Mvc.JsonResult(
                        new ApiResult
                        {
                            Code = (int)ResultCode.TokenIllegal,
                            Message = "用户凭证失效，请重新登录2",
                            Data = ""
                        });
                }
                catch (Exception ex)
                {
                    VTX.FW.Helper.LogHelper.WriteError("JWTValidat3", ex: ex);
                    actionContext.Result = new Microsoft.AspNetCore.Mvc.JsonResult(
                        HttpStatusCode.OK,
                        new ApiResult
                        {
                            Code = (int)ResultCode.TokenIllegal,
                            Message = "用户凭证失效，请重新登录3",
                            Data = ""
                        });
                }
            }
            else
            {
                actionContext.Result = new Microsoft.AspNetCore.Mvc.JsonResult(
                        new ApiResult
                        {
                            Code = (int)ResultCode.TokenIllegal,
                            Message = "用户凭证为空，请重新登录4",
                            Data = ""
                        });
            }
        }
    }
}