﻿
using EduSpider.IRepository;
using EduSpider.IServices;
using EduSpider.Model.Entity;
using EduSpider.Model.Extend;
using EduSpider.Model.Query;
using EduSpider.Repository;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VTX.FW.Attr;

namespace EduSpider.Services
{
    /// <summary>
    /// 课程服务实现
    /// </summary>
    public class CourseService : ICourseService
    {
        /// <summary>
        /// 学员课程仓储接口
        /// </summary>
        [Autowired]
        public ICourseStudentRepository CourseStudentRepository { get; set; }

        /// <summary>
        /// 学员家庭作业仓储接口
        /// </summary>
        [Autowired]
        public IStuHomeWorkRepository StuHomeWorkRepository { get; set; }

        /// <summary>
        /// 评语仓储层接口
        /// </summary>
        [Autowired]
        public ICommentRepository CommentRepository { get; set; }

        /// <summary>
        /// 老师课程关联仓储接口
        /// </summary>
        [Autowired]
        public ICourseTeacherRepository CourseTeacherRepository { get; set; }

        /// <summary>
        /// 作业仓储接口
        /// </summary>
        [Autowired]
        public IHomeWorkRepository HomeWorkRepository { get; set; }

        /// <summary>
        /// 学生考试仓储接口
        /// </summary>
        [Autowired]
        public IExam_ScoreRepository Exam_ScoreRepository { get; set; }

        /// <summary>
        /// 课程评价配置仓储接口
        /// </summary>
        [Autowired]
        public ICourseCommentRepository CourseCommentRepository { get; set; }

        /// <summary>
        /// 课程评价详情配置仓储接口
        /// </summary>
        [Autowired]
        public ICourseCommentDetailsRepository CourseCommentDetailsRepository { get; set; }

        /// <summary>
        /// 学员评价仓储层接口
        /// </summary>
        [Autowired]
        public IStuCommentRepository StuCommentRepository { get; set; }



        /// <summary>
        /// 获取学员课程列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<RB_Course_Student_Extend> GetStuCourseList(CourseQuery query)
        {
            var list = CourseStudentRepository.GetCourseStudentListRepository(query);
            return list;
        }

        /// <summary>
        /// 获取老师课程列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<RB_Course_Teacher_Extend> GetTeacherCourseList(CourseQuery query)
        {
            var list = CourseTeacherRepository.GetTeacherCourseListRepository(query);
            return list;
        }

        /// <summary>
        /// 获取学员考试和家庭作业列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<object> GetStuWorkAndExamList(CourseQuery query)
        {
            List<object> list = new();
            List<ExamWorkResult> dataList = new();
            var homeWorkList = StuHomeWorkRepository.GetStuHomeWorkListRepository(query);
            if (homeWorkList != null && homeWorkList.Count > 0)
            {
                dataList.AddRange(homeWorkList);
            }
            var examList = Exam_ScoreRepository.GetExamScoreListRepository(query);
            if (examList != null && examList.Count > 0)
            {
                foreach (var item in examList)
                {
                    decimal score_p = 0;
                    if (item.ExamScore > 0)
                    {
                        score_p = Math.Round(item.TScore / item.ExamScore, 4);
                    }
                    dataList.Add(new ExamWorkResult()
                    {
                        ResultType = 2,
                        Title = item.ExamName,
                        CreateTime = item.ExamCreateTime,
                        Score = item.TScore,
                        Score_p = score_p,
                        ExamRank = item.Rank,
                        ExamRankRate = item.RankRate,
                        ExamId = item.ExamId,
                        ExamScoreId = item.Id,
                    });
                }
            }

            if (dataList != null && dataList.Count > 0)
            {
                dataList = dataList.OrderBy(qitem => qitem.CreateTime).ToList();
                var firstModel = dataList.FirstOrDefault();
                foreach (var item in dataList)
                {
                    object obj = new();
                    //家庭作业
                    if (item.ResultType == 1)
                    {
                        obj = new
                        {
                            item.Score,
                            Score_p = item.Score_p * 100,
                            item.ScoreType,
                            item.ScoreTypeStr,
                            item.Score_Value,
                            item.HomeWorkStatus,
                            item.HomeWorkStatusStr,
                        };
                    }
                    //考试
                    if (item.ResultType == 2)
                    {
                        obj = new
                        {
                            item.Score,
                            Score_p = item.Score_p * 100,
                            item.ExamRank,
                            item.ExamRankRate,
                            item.ExamId,
                            item.ExamScoreId,
                        };
                    }
                    //评语
                    if (item.ResultType == 3)
                    {
                        obj = new
                        {
                            item.Comment
                        };
                    }
                    string subTitle = "";
                    if (item.CreateTime.Year != firstModel.CreateTime.Year)
                    {
                        subTitle = firstModel.CreateTime.Year.ToString();
                    }
                    string createTime = Convert.ToDateTime(item.CreateTime.ToString()).ToString("MM-dd HH:mm");
                    list.Add(new
                    {
                        item.ResultType,
                        item.ResultTypeStr,
                        item.Title,
                        TimeTile = subTitle,
                        CreateTime = createTime,
                        DataObj = obj
                    });
                    firstModel = item;
                }
            }
            return list;
        }

        /// <summary>
        /// 获取课程作业列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<object> GetCourseHomeWork(CourseQuery query)
        {
            List<object> result = new();
            var list = HomeWorkRepository.GetCourseHomeWorkListRepository(query);
            if (list != null && list.Count > 0)
            {
                foreach (var item in list)
                {
                    string str = "";
                    if (item.Status == 0)
                    {
                        str = "未开始";
                    }
                    else if (item.Status == 1)
                    {
                        str = "进行中";
                    }
                    else if (item.Status == 2)
                    {
                        str = "已结束";
                    }
                    result.Add(new
                    {
                        HomeWorkId = item.Homework_id,
                        HomeWorkTitle = item.Homework_title,
                        item.Status,
                        StatusStr = str,
                        StuNum = item.Num,
                        SubmitNum = item.Cnum,
                        ReviewNum = item.Rnum,
                        ExcellentNum = item.Cream_num
                    });
                }
            }
            return result;
        }

        /// <summary>
        /// 获取作业详情
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public object GetHomeWorkDetails(CourseQuery query)
        {
            List<object> result = new();
            var HomeWork= HomeWorkRepository.GetCourseHomeWorkListRepository(query).FirstOrDefault();
            var list = StuHomeWorkRepository.GetStuHomeWorkDetailsListRepository(query);
            if (list != null && list.Count > 0)
            {
                foreach (var item in list)
                {
                    string str = "";
                    if (item.status == 0)
                    {
                        str = "未提交";
                    }
                    else if (item.status == 1)
                    {
                        str = "已提交";
                    }
                    else if (item.status == 2)
                    {
                        str = "已批阅";
                    }
                    result.Add(new
                    {
                        HomeWorkId = item.homework_id,
                        HomeWorkTitle = item.Homework_title,
                        Status = item.status,
                        StatusStr = str,
                        item.student_name,
                        item.student_uid,
                        item.score,
                        item.score_p,
                        add_time = VTX.FW.Helper.ConvertHelper.FormatDate(item.add_time)
                    });
                }
            }
            var obj = new
            {
                HomeWorkTitle = HomeWork?.Homework_title ?? "",
                CourseName = HomeWork?.Course_name ?? "",
                StuNum = HomeWork?.Num ?? 0,
                SubmitNum = HomeWork?.Cnum ?? 0,
                ReadNum = HomeWork?.Rnum ?? 0,
                GoodNum = HomeWork?.Cream_num ?? 0,
                DataList = result
            };
            return obj;
        }

        /// <summary>
        /// 新增修改课程评价配置
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        public bool SetCourseComment(RB_Course_Comment_Extend model)
        {
            return CourseCommentRepository.SetCourseCommentRepository(model);
        }

        /// <summary>
        /// 获取课程评价配置列表
        /// </summary>
        /// <param name="query"></param>
        /// <returns></returns>
        public List<RB_Course_Comment_Extend> GetCourseCommentList(CourseQuery query)
        {
            var list = CourseCommentRepository.GetCourseCommentListRepository(query);
            return list;
        }

        /// <summary>
        /// 根据评价编号删除课程评价配置
        /// </summary>
        /// <param name="Ids"></param>
        /// <returns></returns>
        public bool DeleteCourseComment(string Ids)
        {
            return CourseCommentRepository.DeleteCourseCommentRepository(Ids);
        }

        /// <summary>
        /// 根据编号删除课程评价详情配置
        /// </summary>
        /// <param name="DetailIds"></param>
        /// <returns></returns>
        public bool DeleteCourseCommentDetail(string DetailIds)
        {
            return CourseCommentDetailsRepository.DeleteCourseCommentDetailsRepository(DetailIds);
        }

        /// <summary>
        /// 新增修改学员评价
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        public bool SetStuComment(RB_Stu_Comment model)
        {
            return StuCommentRepository.SetStuCommentRepository(model);
        }
    }
}
