﻿using EduSpider.Model.Cache;
using EduSpider.Repository;
using EduSpider.Utility;
using EduSpider.Utility.Enum;
using System;
using System.Linq;
using VTX.FW.Helper;
using VTX.FW.Api;
using VTX.FW.Cache;

namespace EduSpider.Cache.User
{
    /// <summary>
    /// redis缓存
    /// </summary>
    public class UserReidsCache
    {
        readonly static RedisCache redis = new(new RedisConfig
        {
            DbNum = 3,
            RedisServer = ConfigHelper.GetAppsettings("RedisSetting", "RedisServer"),
            RedisPort = Convert.ToInt32(ConfigHelper.GetAppsettings("RedisSetting", "RedisPort")),
            RedisPwd = ConfigHelper.GetAppsettings("RedisSetting", "RedisPwd")
        });

        /// <summary>
        /// 设置缓存
        /// </summary>
        /// <param name="model"></param>
        public static void UserInfoSet(string cacheKey, UserInfo model, int JwtExpirTime)
        {
            try
            {
                TimeSpan ts = GetExpirTime(JwtExpirTime);
                redis.StringSet<UserInfo>(cacheKey, model, ts);
            }
            catch (Exception ex)
            {
                LogHelper.WriteError("UserInfoSet", "缓存设置失败", ex);
            }
        }

        /// <summary>
        /// 获取缓存时长
        /// </summary>
        /// <param name="JwtExpirTime"></param>
        /// <returns></returns>
        private static TimeSpan GetExpirTime(int JwtExpirTime)
        {
            DateTime dt = DateTime.Now;
            DateTime dt2 = DateTime.Now;
            TimeSpan ts = dt.AddSeconds(JwtExpirTime) - dt2;
            return ts;
        }

        /// <summary>
        /// 判断key是否存在
        /// </summary>
        /// <param name="cacheKey"></param>
        /// <returns></returns>
        public static bool Exists(string cacheKey)
        {
            return redis.KeyExists(cacheKey);
        }

        /// <summary>
        /// 设置缓存
        /// </summary>
        /// <param name="cacheKey"></param>
        /// <param name="Data"></param>
        /// <param name="JwtExpirTime"></param>
        public static void Set(string cacheKey, object Data, int JwtExpirTime)
        {
            try
            {
                TimeSpan ts = GetExpirTime(JwtExpirTime);
                redis.StringSet(cacheKey, Data, ts);
            }
            catch (Exception)
            {
            }
        }
 
        /// <summary>
        /// 获取缓存
        /// </summary>
        /// <param name="cacheKey"></param>
        public static object Get(string cacheKey)
        {
            try
            {
                object info = redis.StringGet<object>(cacheKey);
                return info;
            }
            catch (Exception)
            {
                return "";
            }
        }

        /// <summary>
        /// 账号仓储层对象
        /// </summary>
        private static readonly AccountRepository accountRepository = new();

        /// <summary>
        /// 获取用户登录信息
        /// </summary>
        /// <param name="Id">账号Id</param>
        /// <param name="apiRequestFromEnum">请求来源</param>
        /// <returns></returns>
        public static UserInfo GetUserLoginInfo(object Id, ApiRequestFromEnum apiRequestFromEnum = ApiRequestFromEnum.MiniProgram)
        {

            UserInfo userInfo = null;
            if (Id != null)
            {
                string cacheKey = CacheKey.User_Login_Key + Id.ToString();
                try
                {
                    userInfo = redis.StringGet<UserInfo>(cacheKey);
                }
                catch (Exception ex)
                {
                    LogHelper.WriteError("GetUserLoginInfo", "获取缓存失败", ex);
                }
                if (userInfo == null)
                {
                    Int32.TryParse(Id.ToString(), out int NewId);
                    if (NewId > 0)
                    {
                        string token = "";
                        var model = accountRepository.GetAccountList(new Model.Extend.rb_account_hk_Extend()
                        {
                            Id = NewId
                        })?.FirstOrDefault();
                        if (model != null)
                        {
                            userInfo = new UserInfo
                            {
                                Id = model.Id,
                                AccountType = model.AccountType,
                                AccountName = model.AccountName,
                                Token = token,
                                ApiRequestFromEnum = apiRequestFromEnum,
                                AccountId = model.AccountId,
                                UserMobile = model.Account,
                                UserIcon = model.UserIcon,
                            };
                            UserInfoSet(CacheKey.User_Login_Key + Id.ToString(), userInfo, Config.JwtExpirTime);
                        }
                    }
                }
            }
            else
            {
                userInfo = new UserInfo();
            }
            return userInfo;
        }

    }
}